<?php

declare(strict_types=1);

namespace Drupal\slots\Plugin\Condition;

use Drupal\Core\Condition\Attribute\Condition;
use Drupal\Core\Condition\ConditionPluginBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\slots\SlotIdMachineNameTrait;
use Drupal\slots\SlotsServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Slot' condition.
 */
#[Condition(
  id: "slot",
  label: new TranslatableMarkup("Slot"),
  context_definitions: [
    "slot_id" => new ContextDefinition(
      data_type: "slot",
      label: new TranslatableMarkup("The slot id"),
    ),
  ]
)]
class SlotCondition extends ConditionPluginBase implements ContainerFactoryPluginInterface, SlotConditionInterface {

  use SlotIdMachineNameTrait;

  /**
   * Constructs a new SlotCondition object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\slots\SlotsServiceInterface $slotsService
   *   The slots service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly SlotsServiceInterface $slotsService,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('slots.service'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return array_merge(parent::defaultConfiguration(), [
      'slot_id' => '',
      'weight' => 0,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['slot_id'] = [
      '#type' => $this->moduleHandler->moduleExists('tagify') ? 'select_tagify' : 'select',
      '#title' => $this->t('Slot identifier'),
      '#options' => $this->slotsService->getSlotIds(),
      '#default_value' => $this->configuration['slot_id'],
      '#empty_option' => $this->t('- Select a slot -'),
      '#required' => TRUE,
    ];

    if ($this->moduleHandler->moduleExists('tagify')) {
      $form['slot_id']['#type'] = 'select_tagify';
      $form['slot_id']['#identifier'] = $form['#id'];
      $form['slot_id']['#attributes'] = ['class' => [$form['#id']]];
    }

    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $this->configuration['weight'],
      '#delta' => 10,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['slot_id'] = $form_state->getValue('slot_id');
    $this->configuration['weight'] = $form_state->getValue('weight');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    if (!empty($this->configuration['negate'])) {
      return $this->t('Do not apply for slots with the id: %id', ['%id' => $this->configuration['slot_id']]);
    }

    return $this->t('Apply for slots with the id: %id', ['%id' => $this->configuration['slot_id']]);
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $slot_id = $this->getContext('slot_id')->getContextValue();
    if (!empty($this->configuration['negate'])) {
      return $this->configuration['slot_id'] !== $slot_id;
    }

    return $this->configuration['slot_id'] === $slot_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return (int) $this->configuration['weight'];
  }

  /**
   * {@inheritdoc}
   */
  public function getSlotId(): string {
    return $this->configuration['slot_id'];
  }

}
