<?php

declare(strict_types=1);

namespace Drupal\slots\Plugin\Field\FieldWidget;

use Drupal\Core\Field\WidgetBase;
use Drupal\slots\SlotsServiceInterface;
use Drupal\Component\Serialization\Json;
use Drupal\conditions\ConditionsServiceInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the 'slot_content_weights' field widget.
 */
#[FieldWidget(
  id: 'slot_content_weights',
  label: new TranslatableMarkup('Slot content weights'),
  field_types: ['dynamic_entity_reference'],
  multiple_values: TRUE,
)]
final class SlotContentWeightsWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    protected readonly SlotsServiceInterface $slotsService,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly ConditionsServiceInterface $conditionsService,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($plugin_id, $plugin_definition, $configuration['field_definition'], $configuration['settings'], $configuration['third_party_settings'],
      $container->get('slots.service'),
      $container->get('entity_type.manager'),
      $container->get('conditions.service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    // Disallow this widget for standard entity reference fields.
    if (isset($field_definition->getDisplayOptions('form')['type'])) {
      return $field_definition->getDisplayOptions('form')['type'] == 'slot_content_weights';
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element['#type'] = 'fieldset';
    $element['#description'] = $this->t('You can change the order of the content by dragging it into the desired position.');

    $element['weights'] = [
      '#parent' => 'container',
      '#type' => 'table',
      '#header' => [$this->t('Content'), $this->t('Weight')],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'item-weight',
        ],
      ],
    ];

    foreach ($items as $delta => $item) {
      if (!$item->entity) {
        continue;
      }
      $element['weights'][$delta]['#attributes']['class'][] = 'draggable';
      $element['weights'][$delta]['label'] = $item->entity->toLink()->toRenderable();
      $element['weights'][$delta]['weight'] = [
        '#type' => 'weight',
        '#title_display' => 'invisible',
        '#default_value' => $item->weight,
        '#attributes' => ['class' => ['item-weight']],
      ];
      $element['weights'][$delta]['entity_id'] = [
        '#type' => 'hidden',
        '#default_value' => $item->entity->id(),
      ];
      $element['weights'][$delta]['entity_type'] = [
        '#type' => 'hidden',
        '#default_value' => $item->entity->getEntityTypeId(),
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Use this method to save weights on the content entities' slot conditions.
    foreach ($values['weights'] as $row) {
      /** @var \Drupal\Core\Entity\FieldableEntityInterface $slot_content */
      $slot_content = $this->entityTypeManager->getStorage($row['entity_type'])->load($row['entity_id']);
      $this->setWeight($slot_content, $form_state->getFormObject()->getEntity()->id(), (int) $row['weight']);
    }

    return parent::massageFormValues($values, $form, $form_state);
  }

  /**
   * Set the weight on a slot content entity.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The slot content entity.
   * @param string $slot_id
   *   The slot_id to set the weight for.
   * @param int|float $weight
   *   The weight.
   */
  protected function setWeight(FieldableEntityInterface $entity, string $slot_id, int|float $weight) {
    foreach ($entity->getFields(FALSE) as $field) {
      if ($field->getFieldDefinition()->getType() != 'slots') {
        continue;
      }
      $condition_groups = [];
      foreach ($field->getValue() as $delta => $condition_group) {
        $conditions = [];
        foreach (Json::decode($condition_group['value']) as $uuid => $condition) {
          $conditions[$uuid] = $condition;
          if ($condition['id'] == 'slot' && $condition['slot_id'] == $slot_id) {
            $conditions[$uuid]['weight'] = $weight;
          }
        }

        $condition_group['value'] = Json::encode($conditions);
        $condition_groups[$delta] = $condition_group;
      }

      $field->setValue($condition_groups);

      $entity->save();
    }
  }

}
