<?php

declare(strict_types=1);

namespace Drupal\slots\Plugin\views\field;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Url;
use Drupal\views\ResultRow;
use Drupal\views\Attribute\ViewsField;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\slots\Plugin\Field\FieldType\SlotsItemInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Exception\RouteNotFoundException;

/**
 * Provides slot ID field handler.
 */
#[ViewsField('slot_id')]
class SlotId extends FieldPluginBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('router.route_provider'),
      $container->get('redirect.destination'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly RouteProviderInterface $routeProvider,
    protected readonly RedirectDestinationInterface $redirectDestination,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $entity = $this->getEntity($values);
    if (!$entity instanceof ContentEntityInterface) {
      return [
        '#type' => 'markup',
        '#markup' => $this->t('- None -'),
      ];
    }
    $build = [
      '#theme' => 'item_list',
      '#items' => [],
    ];
    foreach ($entity->getFields(FALSE) as $field) {
      if ($field->getFieldDefinition()->getType() != 'slots') {
        continue;
      }

      foreach ($field as $item) {
        if (!$item instanceof SlotsItemInterface) {
          continue;
        }
        if (!$item->getSlotsStatus()) {
          continue;
        }
        $conditions = Json::decode($item->value);
        foreach ($conditions as $condition) {
          if ($condition['id'] != 'slot') {
            continue;
          }
          try {
            $this->routeProvider->getRouteByName('view.slot_content.page');
            $build['#items'][] = [
              '#type' => 'link',
              '#title' => $condition['slot_id'],
              '#url' => Url::fromRoute(
                'entity.slot.canonical',
                ['slot' => $condition['slot_id']],
                ['query' => $this->redirectDestination->getAsArray()]
              ),
            ];
          }
          catch (RouteNotFoundException $e) {
            $build['#items'][] = $condition['slot_id'];
          }
        }
      }
    }

    $cacheable_metadata = CacheableMetadata::createFromObject($entity);
    $cacheable_metadata->applyTo($build);

    return $build;
  }

}
