<?php

namespace Drupal\slots\Plugin\views\filter;

use Drupal\conditions_field\ConditionsFieldServiceInterface;
use Drupal\Core\Database\Connection;
use Drupal\slots\SlotsServiceInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\filter\StringFilter;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Simple filter to handle greater than/less than filters.
 *
 * @ingroup views_filter_handlers
 */
#[ViewsFilter("slot_id_filter")]
class SlotIdFilter extends StringFilter implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new LatestRevision.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\slots\SlotsServiceInterface $slotsService
   *   The slots service.
   * @param \Drupal\conditions_field\ConditionsFieldServiceInterface $conditionsFieldService
   *   The conditions field service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    Connection $connection,
    protected readonly SlotsServiceInterface $slotsService,
    protected readonly ConditionsFieldServiceInterface $conditionsFieldService,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $connection);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition,
      $container->get('database'),
      $container->get('slots.service'),
      $container->get('slots.conditions_field.service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildExposedForm(&$form, FormStateInterface $form_state) {
    parent::buildExposedForm($form, $form_state);
    if (!empty($this->options['expose']['identifier'])) {
      $value = $this->options['expose']['identifier'];
      $form[$value]['#type'] = 'select';
      $form[$value]['#options'] = $this->slotsService->getSLotIds();
      $form[$value]['#empty_option'] = $this->t('- All -');
      $form[$value]['#size'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    if ($slot_id = $this->value) {
      $this->ensureMyTable();
      $field = "$this->tableAlias.$this->realField";
      // Get all block content entities that have the slot id condition.
      $items = $this->conditionsFieldService->getItemsDataByConditionConfiguration(
        $this->getEntityType(),
        [
          'id' => 'slot',
          'slot_id' => $slot_id,
        ],
        [
          'status' => '1',
          'slots_status' => '1',
        ],
      );
      $entity_ids = array_keys($items);

      // If no entity was found add a condition that can not be fulfilled.
      if (empty($entity_ids)) {
        $entity_ids = [-1];
      }
      /** @var \Drupal\views\Plugin\views\query\Sql $query */
      $query = $this->query;
      $query->addWhere($this->options['group'], $field, $entity_ids, 'IN');
    }
  }

}
