<?php

declare(strict_types=1);

namespace Drupal\slots;

use Drupal\Core\Database\Connection;
use Drupal\Component\Utility\SortArray;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Plugin\Context\Context;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Render\RenderableInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\conditions\ConditionsServiceInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Condition\ConditionAccessResolverTrait;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\slots\Plugin\Condition\SlotConditionInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\conditions_field\ConditionsFieldServiceInterface;
use Drupal\block_plugin_view_builder\BlockPluginViewBuilderInterface;

/**
 * The slots service.
 */
class SlotsService implements SlotsServiceInterface {

  use ConditionAccessResolverTrait;
  use StringTranslationTrait;

  /**
   * The slot entity types.
   */
  protected array $slotEntityTypes = [];

  /**
   * The slot bundles.
   */
  protected array $slotBundles = [];

  /**
   * Slot content static cache.
   */
  protected array $slotContents = [];

  /**
   * Constructs a SlotsService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $conditionManager
   *   The plugin.manager.condition service.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $contextRepository
   *   The context repository service.
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $contextHandler
   *   The plugin context handler.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection to be used.
   * @param \Drupal\conditions_field\ConditionsFieldServiceInterface $conditionsFieldService
   *   The conditions field service.
   * @param \Drupal\block_plugin_view_builder\BlockPluginViewBuilderInterface $blockPluginViewBuilder
   *   The block plugin view builder service.
   * @param \Drupal\conditions\ConditionsServiceInterface $conditionsService
   *   The conditions service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   A cache backend interface.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly AccountInterface $currentUser,
    protected readonly PluginManagerInterface $conditionManager,
    protected readonly ContextRepositoryInterface $contextRepository,
    protected readonly ContextHandlerInterface $contextHandler,
    protected readonly Connection $database,
    protected readonly ConditionsFieldServiceInterface $conditionsFieldService,
    protected readonly BlockPluginViewBuilderInterface $blockPluginViewBuilder,
    protected readonly ConditionsServiceInterface $conditionsService,
    protected readonly CacheBackendInterface $cache,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getSlotContext(string $slot_id, string $context_id = '@slots.context_provider:slot'): array {
    $context = new Context(new ContextDefinition('slot', $this->t('The slot.')), $slot_id);
    return [
      $context_id => $context,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildSlot(string $slot_id, int $cardinality = 0, string $label = ''): RenderableInterface {
    $build = [];
    $build['#cache']['tags'][] = 'slot:' . $slot_id;

    // Ensure the slot entity exists and, if a non-empty label is provided,
    // persist it on first creation or when the existing label is still the
    // default (equal to the ID). This allows labels coming from Paragraphs or
    // Views to be saved when the slot is "collected" during rendering.
    $this->createSlot($slot_id, $label);

    $slot = $this->blockPluginViewBuilder->view('slot_block', [
      'slot_id' => $slot_id,
      // Do not pass a label to the block render to avoid it becoming the block
      // title in the UI. Labels are persisted via createSlot() above.
      'cardinality' => $cardinality,
    ]);
    $build[] = $slot;

    $length = $this->buildSlotContents($slot_id)['#length'];
    // We don't want to hide the slot IDs for empty slots.
    if ($this->currentUser->hasPermission('view slot identifiers')) {
      $length++;
    }

    return new Slot($build, $length);
  }

  /**
   * {@inheritdoc}
   */
  public function buildSlotContents(string $slot_id): array {
    if (isset($this->slotContents[$slot_id])) {
      return $this->slotContents[$slot_id];
    }

    $build = ['#length' => 0];

    // Find out which potential cache metadata needs to be taken care of.
    $cacheable_metadata = new CacheableMetadata();
    foreach ($this->getSlotEntityTypes() as $entity_type_id => $entity_type) {
      if ($condition_collections = $this->getMatchingConditions($slot_id, $entity_type_id)) {
        foreach ($condition_collections as $conditions) {
          foreach ($conditions as $condition) {
            $cacheable_metadata->addCacheableDependency($condition);
          }
        }
      }
      $cacheable_metadata->applyTo($build);
      // Add slot_id cache tag, so we can clear the cache for every slot
      // precisely.
      $build['#cache']['tags'][] = 'slot:' . $slot_id;

      // Add the blocks we found to the render array.
      if ($matching_items = $this->getMatchingItemsBySlotId($slot_id, $entity_type_id)) {
        foreach ($matching_items as $entity_id => $metadata) {
          $block_content_display = $this->blockPluginViewBuilder->view(
            'slot_content_block',
            [
              'entity_type_id' => $entity_type_id,
              'entity_id' => $entity_id,
            ]
          );
          $build[] = [
            'content' => $block_content_display,
            '#weight' => $this->getWeightByConditionGroups($metadata['condition_groups'], $slot_id),
          ];
          $build['#length']++;
        }
      }

      uasort($build, [SortArray::class, 'sortByWeightProperty']);
    }

    $this->slotContents[$slot_id] = $build;

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getSlotEntityTypes(): array {
    if (empty($this->slotEntityTypes)) {
      foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
        $conditions_fields = $this->conditionsFieldService->getConditionsFieldDefinitions($entity_type_id);
        if (!empty($conditions_fields)) {
          $this->slotEntityTypes[$entity_type_id] = $entity_type;
        }
      }
    }

    return $this->slotEntityTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function getMatchingItemsBySlotId(string $slot_id, string $entity_type_id): ?array {
    $matching_items = $this->conditionsFieldService->getMatchingItems(
      $entity_type_id,
      [
        'id' => 'slot',
        'slot_id' => $slot_id,
      ],
      $this->getSlotContext($slot_id),
      [
        'status' => '1',
        'slots_status' => '1',
      ],
    );
    return !empty($matching_items) ? $matching_items : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getMatchingConditions(string $slot_id, string $entity_type_id): ?array {
    // @todo Unify with ConditionsFieldService::getMatchingItems()
    // Load all potential items from database.
    $matching_items = $this->conditionsFieldService->getItemsDataByConditionConfiguration(
      $entity_type_id,
      [
        'id' => 'slot',
        'slot_id' => $slot_id,
      ],
      [
        'status' => '1',
        'slots_status' => '1',
      ],
    );

    // Build the conditions used in the slots field.
    $condition_collections = [];
    foreach ($matching_items as $condition_groups) {
      // We need to initialize the conditions in order to get their cacheable
      // metadata.
      $this->conditionsService->initializeConditions($condition_groups, []);
      foreach ($condition_groups as $condition_group) {
        $condition_collections[] = $condition_group['conditions'];
      }
    }

    return !empty($condition_collections) ? $condition_collections : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function entityHasSlotCondition(EntityInterface $entity): bool {
    $condition_fields = $this->conditionsFieldService->getConditionsFieldDefinitions($entity->getEntityTypeId());
    foreach ($condition_fields as $field_name => $field_definition) {
      // Make sure the bundle of our entity uses the field.
      if ($entity->hasField($field_name)) {
        foreach ($entity->get($field_name) as $field_condition) {
          // The conditions field stores the conditions in a json field.
          $conditions = !empty($field_condition->value) ? JSON::decode($field_condition->value) : [];
          foreach ($conditions as $condition) {
            // If a condition with the id "slot" exists, we return TRUE.
            if (isset($condition['id']) && $condition['id'] == 'slot') {
              return TRUE;
            }
          }
        }
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function resetCaches(): void {
    $this->slotContents = [];
    $this->slotEntityTypes = [];
  }

  /**
   * {@inheritdoc}
   */
  public function createSlot(string $slot_id, string $label = ''): ?int {
    $slot_storage = $this->entityTypeManager->getStorage('slot');
    /** @var \Drupal\slots\SlotInterface|null $existing */
    $existing = $slot_storage->load($slot_id);
    if (!$existing) {
      /** @var \Drupal\slots\SlotInterface $slot */
      $slot = $slot_storage->create([
        'id' => $slot_id,
        'label' => $label !== '' ? $label : $slot_id,
      ]);
      return $slot->save();
    }

    // If the slot already exists and a non-empty label is provided, update the
    // label only when the existing label appears to be the default (either
    // empty or equal to the ID). This prevents overwriting intentionally set
    // labels when rendering collects slots without a label.
    if ($label !== '') {
      $current_label = (string) $existing->label();
      if ($current_label === '' || $current_label === $existing->id()) {
        $existing->set('label', $label);
        return $existing->save();
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getSlotIds(): array {
    if ($cached = $this->cache->get('slot_ids')) {
      return $cached->data;
    }

    $slot_ids = [];
    $slot_storage = $this->entityTypeManager->getStorage('slot');
    $query = $slot_storage->getQuery()->accessCheck(FALSE);
    foreach ($query->execute() as $slot_id) {
      $slot = $slot_storage->load($slot_id);
      $slot_ids[$slot->id()] = $slot->label();
    }

    // Sort slot IDs naturally and case insensitive.
    natcasesort($slot_ids);

    $this->cache->set('slot_ids', $slot_ids, tags: ['slot_list']);
    return $slot_ids;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeightByConditionGroups(array $condition_groups, string $slot_id) : ?int {
    foreach ($condition_groups as $condition_group) {
      foreach ($condition_group['conditions'] ?? [] as $condition) {
        if (!$condition instanceof SlotConditionInterface) {
          continue;
        }
        if ($condition->getSlotId() != $slot_id) {
          continue;
        }
        return $condition->getWeight();
      }
    }

    return NULL;
  }

}
