<?php

declare(strict_types=1);

namespace Drupal\Tests\slots\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Component\Serialization\Json;
use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;

/**
 * Test the slots module.
 *
 * @group slots
 */
final class SlotsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'slots',
    'block',
    'block_content',
    'system',
    'user',
    'conditions',
    'conditions_field',
    'block_plugin_view_builder',
    'field',
    'json_field',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('block_content');
    $this->installEntitySchema('slot');
    $this->installConfig(['field']);
  }

  /**
   * Test content is determined by SlotService::buildSlotContents() correctly.
   */
  public function testBuildSlotContents(): void {
    // Create a block content type.
    BlockContentType::create([
      'id' => 'spiffy',
      'label' => 'Mucho spiffy',
      'description' => "Provides a block type that increases your site's spiffiness by up to 11%",
    ])->save();

    $fieldStorage = \Drupal::entityTypeManager()->getStorage('field_storage_config')->create([
      'field_name' => 'field_slots',
      'entity_type' => 'block_content',
      'type' => 'slots',
    ]);
    $fieldStorage->save();
    $field = FieldConfig::create([
      'field_storage' => $fieldStorage,
      'label' => 'Slots',
      'bundle' => 'spiffy',
      'description' => 'Description for slots',
      'required' => FALSE,
    ]);
    $field->save();

    // And a block content entity.
    $slot_content = BlockContent::create([
      'info' => 'Spiffy prototype',
      'type' => 'spiffy',
    ]);
    $slot_content->set('field_slots', [
      'condition_logic' => 'and',
      'group_condition_logic' => 'or',
      'value' => '{"9546f5e5-010a-4a5b-9a72-52fd020373a8":{"id":"slot","uuid":"9546f5e5-010a-4a5b-9a72-52fd020373a8","slot_id":"not_my_slot","negate":false,"context_mapping":{"slot_id":"@slots.context_provider:slot"}}}',
      'slots_status' => 1,
      'status' => 1,
    ]);
    $slot_content->save();

    // And a block content entity.
    $slot_content = BlockContent::create([
      'info' => 'Spiffy prototype',
      'type' => 'spiffy',
    ]);
    $slot_content->set('field_slots', [
      'condition_logic' => 'and',
      'group_condition_logic' => 'or',
      'value' => '{"9546f5e5-010a-4a5b-9a72-52fd020373a9":{"id":"slot","uuid":"9546f5e5-010a-4a5b-9a72-52fd020373a9","slot_id":"my_slot","negate":false,"context_mapping":{"slot_id":"@slots.context_provider:slot"}}}',
      'slots_status' => 1,
      'status' => 1,
    ]);
    $slot_content->save();

    /** @var \Drupal\slots\SlotsServiceInterface $slots_service */
    $slots_service = \Drupal::service('slots.service');
    $build = $slots_service->buildSlotContents('my_slot');
    $lazy['entity_id'] = NULL;
    $i = 0;
    foreach ($build as $item) {
      if (!is_array($item) || !array_key_exists('content', $item)) {
        continue;
      }

      $i++;
      $lazy = Json::decode($item['content']['slot_content_block']['#lazy_builder'][1][1]);
    }

    self::assertEquals(1, $build['#length'], 'Wrong number of slot content');
    self::assertEquals($slot_content->id(), $lazy['entity_id'], 'Wrong content inside slot');
  }

  /**
   * Test that slot contents are sorted by weight, not creation order.
   */
  public function testBuildSlotContentsSortedByWeight(): void {
    // Create a block content type.
    BlockContentType::create([
      'id' => 'spiffy',
      'label' => 'Mucho spiffy',
    ])->save();

    // Create slots field on block_content.
    $fieldStorage = \Drupal::entityTypeManager()->getStorage('field_storage_config')->create([
      'field_name' => 'field_slots',
      'entity_type' => 'block_content',
      'type' => 'slots',
    ]);
    $fieldStorage->save();
    FieldConfig::create([
      'field_storage' => $fieldStorage,
      'label' => 'Slots',
      'bundle' => 'spiffy',
    ])->save();

    // Create 3 blocks with deliberately shuffled weights.
    $weights = [20, 5, 10];
    $blocks = [];
    foreach ($weights as $index => $weight) {
      $block = BlockContent::create([
        'info' => 'Block ' . $index,
        'type' => 'spiffy',
      ]);
      $block->set('field_slots', [
        'condition_logic' => 'and',
        'group_condition_logic' => 'or',
        'value' => Json::encode([
          'uuid-' . $index => [
            'id' => 'slot',
            'uuid' => 'uuid-' . $index,
            'slot_id' => 'my_slot',
            'weight' => $weight,
            'negate' => FALSE,
            'context_mapping' => ['slot_id' => '@slots.context_provider:slot'],
          ],
        ]),
        'slots_status' => 1,
        'status' => 1,
      ]);
      $block->save();
      $blocks[$weight] = (int) $block->id();
      // Sleep to ensure different created timestamps so order isn't accidental.
      usleep(100000);
    }

    /** @var \Drupal\slots\SlotsServiceInterface $slots_service */
    $slots_service = \Drupal::service('slots.service');
    $build = $slots_service->buildSlotContents('my_slot');

    // Extract entity_ids in the order they were returned.
    $returned_ids = [];
    foreach ($build as $item) {
      if (!is_array($item) || !array_key_exists('content', $item)) {
        continue;
      }
      $lazy = Json::decode($item['content']['slot_content_block']['#lazy_builder'][1][1]);
      $returned_ids[] = (int) $lazy['entity_id'];
    }

    // Sort our expected IDs by weight ascending.
    ksort($blocks);
    $expected_ids = array_values($blocks);

    self::assertSame(
      $expected_ids,
      $returned_ids,
      'Slot contents are not sorted by weight ascending'
    );
  }

}
