<?php

namespace Drupal\smallads;

use Drupal\smallads\Entity\SmalladType;
use Drupal\Core\Breadcrumb\BreadcrumbBuilderInterface;
use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Cache\CacheableMetadata;

/**
 * A class to build breadcrumbs for all node routes.
 * It doesn't do views. Note that the 'news' view is the default home page.
 */
class BreadcrumbBuilder implements BreadcrumbBuilderInterface {
  use StringTranslationTrait;

  private $currentRoute;

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match, ?CacheableMetadata $cacheable_metadata = null) {
    return str_contains($route_match->getRouteName(), 'smallad') and !$route_match->getRouteObject()->getOption('_admin_route');
  }
  /**
   * {@inheritdoc}
   */
  public function build(RouteMatchInterface $route_match) {
//    smallads_murmurations.want.map  Wants > global > map
//    smallads_murmurations.offer.map  Offers > global > map
//    smallads_murmurations.want.list  Wants > global > list
//    smallads_murmurations.offer.list Wants > global > list
//    entity.smallad.canonical Smallads > TITLE
//    entity.smallad.edit_form Smallads > TITLE > edit
//    view.smallads_auto_page.offer > Offers
//    view.smallads_auto_page.want > Wants
//    view.smallads_user.my_ads Users > NAME > Offers & Wants
    $links = [];

    [$p1, $p2, $p3] = array_pad(explode('.', $route_match->getRouteName()), 3, 0);
    if ($route_match->getRouteName() == 'entity.smallad.add_form') {
      $smallad_type = $route_match->getParameter('smallad_type');
      $links[] = Link::createFromRoute($smallad_type->labelPlural(), 'view.smallads_auto_page.'.$smallad_type->id());
      $links[] = Link::createFromRoute($this->t('Create'), '<current>');
    }
    elseif ($p1 == 'entity') {
      $smallad = $route_match->getParameter('smallad');
      $smallad_type = SmalladType::load($smallad->type->target_id);
      $links[] = Link::createFromRoute($smallad_type->labelPlural(), 'view.smallads_auto_page.'.$smallad->type->target_id);
      $links[] = Link::createFromRoute($smallad->getowner()->getDisplayname(), 'view.smallads_user.my_ads', ['user' => $smallad->getOwnerId()]);
      $links[] = Link::createFromRoute($smallad->label(), 'entity.smallad.canonical', ['smallad' => $smallad->id()]);
      if ($p3 == 'edit_form') {
        $links[] = Link::createFromRoute($this->t('Edit'), '<current>');
      }
    }
    elseif ($p1 == 'smallads_murmurations') {
      $links[] = Link::createFromRoute(SmalladType::load($p2)->labelPlural(), 'view.smallads_auto_page.'.$p2);
      $links[] = Link::createFromRoute($this->t('Global'), '<current>');
      if ($p3 == 'map') {
        $links[] = Link::createFromRoute($this->t('Map'), '<current>');
      }
      elseif ($p3 == 'list') {
        $links[] = Link::createFromRoute($this->t('List'), '<current>');
      }
    }
    elseif ($p2 == 'smallads_auto_page') {
      $links[] = Link::createFromRoute($this->getViewTitle($p2, $p3), '<current>');
    }
    elseif ($route_match->getParameter('node')) {
      $links = $this->nodeLinks($route_match);
    }
    else {
      $links[] = Link::createFromRoute($route_match->getRouteName(), '<current>');
    }
    return $this->breadcrumbs($links)->addCacheTags(['node:'.$route_match->getRawparameter('node')]);
  }

  /**
   * Create the last links for node view/edit page.
   *
   * @param RouteMatchInterface $route_match
   */
  protected function nodeLinks(RouteMatchInterface $route_match) : array {
    $links = [];
    // Final link is the name of the node
    $node = $route_match->getParameter('node');
    if ($route_match->getRouteName() == 'entity.node.edit_form') {
      $links[] = Link::createFromRoute($node->label(), 'entity.node.canonical', ['node' => $node->id()]);
      $links[] = Link::createFromRoute($this->t('Edit'), '<current>');
    }
    elseif ($route_match->getRouteName() == 'entity.node.canonical') {
      $links[] = Link::createFromRoute($node->label(), '<current>');
    }
    return $links ?? [];
  }

  /**
   * Prefix the home link and convert the link array to a set of breadcrumbs.
   *
   * @param array $links
   * @return type
   */
  protected function breadcrumbs(array $links) : Breadcrumb {
    array_unshift($links, Link::createFromRoute($this->t('Home'), '<front>'));
    return (new Breadcrumb())
      ->setLinks($links)
      ->addCacheContexts(['url.path.parent', 'url.path.is_front']);
  }

  /**
   * @param $view_id
   * @param $display_id
   * @return string
   */
  private function getViewTitle($view_id, $display_id) :string {
    $view = \Drupal\views\Views::getView($view_id);
    $view->setDisplay($display_id);
    $title = $view->getDisplay()->getOption('title');
    if (!$title) { // Get the default title
      $view->setDisplay('default');
      $title = $view->getDisplay()->getOption('title');
      if (!$title) {
        // @todo No title for view smallads_user display my_ads
        // this views display doesn't have a title because it is generated automatically
        \Drupal::logger('breadcrumbs')->warning('No title for view %view display %display', ['%view' => $view_id, '%display' => $display_id]);
        $title = 'Untitled!!';
      }
    }
    return $title;
  }

}
