<?php

namespace Drupal\smallads;

use Drupal\smallads\Entity\SmalladInterface;
use Drupal\user\Entity\User;
use Drupal\file\Entity\File;
use Drupal\taxonomy\Entity\Term;
use Drupal\migrate\MigrateSkipRowException;
use Drupal\migrate\Event\MigratePreRowSaveEvent;
use Drupal\migrate\Event\MigrateImportEvent;
use Drupal\migrate\Event\MigratePostRowSaveEvent;
use Drupal\migrate\Plugin\migrate\destination\EntityContentBase;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Modify migrations before saving them.
 *
 * This is necessary for the links and more importantly the autogenerated tasks
 * Seems like a shortcoming in the routing system, but it will certainly do for
 * this module.
 */
class MigrationSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() : array {
    //Migrate module might not be installed and we can't test for it here because
    //\Drupal::moduleHandler()->moduleExists('migrate_drupal') is not initialised yet!
    //so just replace the constants with their strings defined in Drupal\migrate\Event\MigrateEvents
    return [
      'migrate.pre_import' => ['migratePreImport'],
      'migrate.pre_row_save' => ['migratePreRowSave'],
      //$instances['d7_mc_wallet']->getSourcePlugin()->count()'migrate.post_row_save' => [['migratePostRowSave']]
      'migrate.post_import' => ['migratePostImport'],
    ];
  }

  /**
   * {@inheritDoc}
   */
  function migratePreImport(MigrateImportEvent $event) {
    $migration = $event->getMigration();
    if ($migration->id() == 'd7_taxonomy_term:offers_wants_categories') {
      // delete all the existing categories before migrating old ones.
      foreach (Term::loadMultiple() as $term) {
        if ($term->vid->target_id == 'categories') {
          $term->delete();
        }
      }
    }
  }

  /**
   * Change the field names.
   */
  function migratePreRowSave(MigratePreRowSaveEvent $event) {
    $row = $event->getRow();
    $migration = $event->getMigration();
    $source = $row->getSource();

    if($migration->id() == 'd7_comment' and $row->getDestinationProperty('entity_type') == 'smallads') {
      $row->setDestinationProperty('comment_type', 'smallad');
      $row->setDestinationProperty('field_name', 'comments');
      $row->setDestinationProperty('body', $row->getDestinationProperty('comment_body'));// Assumes no comments on transactions
    }
    if($migration->id() == 'd7_comment_type') {
      if ($row->getSourceProperty('type') == 'proposition') {
        throw new MigrateSkipRowException();
      }
    }
    if($migration->id() == 'd7_comment_field') {
      if ($row->getSourceProperty('type') == 'proposition') {
        throw new MigrateSkipRowException();
      }
    }

    // This is horrible but necessary since the entity name changed
    if ($plugin = $migration->getDestinationConfiguration()['plugin']) {
      if ($plugin == 'component_entity_display' or $plugin == 'component_entity_form_display') {
        if ($row->getSourceProperty('bundle') == 'proposition') {
          throw new MigrateSkipRowException();
        }
        if($row->getDestinationProperty('formatter_type') == 'shs_default') { // source was taxonomy_shs
          $row->setDestinationProperty('options/type', 'entity_reference_label');
          $row->setDestinationProperty('formatter_type', 'entity_reference_label');
        }
      }
      elseif ($row->getDestinationProperty('bundle') == 'proposition') {
        $row->setDestinationProperty('entity_type', 'smallad');
        $row->setDestinationProperty('bundle', 'offer'); // what happened to want?
      }
    }

    // Entities may have re-used the offers_wants_catories field.
    // If any entity used the categories field in d7, change the name of the field in this row
    if ($migration->getDestinationPlugin() instanceof EntityContentBase) {
      if ($row->hasSourceProperty('offers_wants_categories')) {
        $tids = [];
        foreach ($row->getSourceProperty('offers_wants_categories') as $delta => $val) {
          $tids[$delta]['target_id'] = $val['tid'];
        }
        $row->setDestinationProperty('categories', $tids);
        #$row->setDestinationProperty('category', $tids);
        $row->removeDestinationProperty('offers_wants_categories');
      }
    }

    if ($migration->getBaseId() == 'd7_smallad') {
      $uid = $row->getSourceProperty('node_uid');
      if (!User::load($uid)) {
        throw new MigrateSkipRowException();
      }
      // If the expiry date is before now, set the scope to private
      elseif ($row->getDestinationProperty('expires') < \Drupal::Time()->getRequestTime()) {
        $row->setDestinationProperty('scope', SmalladInterface::SCOPE_PRIVATE);
      }
      $row->setDestinationProperty('default_langcode', 1);
    }

    switch ($migration->id()) {
      case 'd7_block':
        if ($row->getSourceProperty('delta') == 'add_proposition') {
          $row->setDestinationProperty('plugin', 'add_smallad');
        }
        elseif ($row->getSourceProperty('delta') == 'propositions_taxonomy_block') {
          $row->setDestinationProperty('id', $row->getDestinationProperty('theme') .'_smallad_categories_nested');
          $row->setDestinationProperty('settings/label', 'Ad categories');
        }
        break;
      case 'd7_taxonomy_vocabulary':
        $vocab = $row->getSourceProperty('machine_name');
        // Delete existing terms in categories vocab.
        if ($vocab == 'offers_wants_categories') {
          $termStorage = \Drupal::entityTypeManager()->getStorage('taxonomy_term');
          foreach ($termStorage->loadByProperties(['vid' => 'categories']) as $term) {
            $term->delete();
          }
          throw new MigrateSkipRowException();
        }
        elseif ($vocab == 'offers_wants_types') {
          throw new MigrateSkipRowException();
        }
        break;
      case 'd7_taxonomy_term:offers_wants_categories':
        $row->setDestinationProperty('vid', 'categories');
        break;
      case 'd7_taxonomy_term:offers_wants_types':
        throw new MigrateSkipRowException();
      case 'd7_field':
        if ($row->getDestinationProperty('entity_type') == 'smallad' and $row->getDestinationProperty('field_name') == 'categories') {
          \Drupal::ConfigFactory()->getEditable('smallad.settings')
            ->set('categories_cardinality', $row->getSourceProperty('cardinality'))
            ->save();
          throw new MigrateSkipRowException();
        }
        break;
      case 'd7_field_instance':
        if (in_array($row->getSourceProperty('field_name'), ['offers_wants_types', 'comment_node_proposition'])) {
          throw new MigrateSkipRowException();
        }
        if ($row->getSourceProperty('field_name') == 'comment_body' and $row->getSourceProperty('entity_type') == 'smallad') {
          throw new MigrateSkipRowException();
        }
        if ($row->getSourceProperty('bundle') == 'comment_node_proposition') {
          throw new MigrateSkipRowException();
        }
        if ($row->getSourceProperty('field_name') == 'offers_wants_categories') {
          if ($row->getDestinationProperty('entity_type') == 'smallad') {
            throw new MigrateSkipRowException();
          }
        }
        break;
      case 'd7_field_instance_widget_settings':
        // this is trying to process without the bundle, so throws errors, but I think gives the required results.
        break;
      case 'd7_taxonomy_vocabulary':
        if ($row->getSourceProperty('machine_name') == 'offers_wants_types') {
          throw new MigrateSkipRowException();
        }
        break;
      case 'd7_node_type':
        // don't migrate old node type
        if ($row->getSourceProperty('type') == 'proposition') {
          throw new MigrateSkipRowException();
        }
        break;
      case 'd7_comment_entity_display':
        if ($row->getSourceProperty('bundle') == 'proposition') {
          throw new MigrateSkipRowException();
        }
        break;
      case 'd7_comment_field':
        if ($row->getSourceproperty('node_type') == 'proposition') {
          throw new MigrateSkipRowException();
        }
        break;
      case 'd7_field_formatter_settings':
        if ($row->getSourceproperty('field_name') == 'offers_wants_categories') {
          $row->setDestinationProperty('field_name', 'category');
        }
        break;
      case 'd7_menu_links':
        $obselete = ['admin/config/ow', 'node/add'];
        if (in_array($row->getSourceProperty('link_path'), $obselete)) {
          throw new MigrateSkipRowException();
        }
        if ($row->getSourceProperty('link_path') == 'node/add/offer') {
          $row->setSourceProperty('link_path', 'ad/add/offer');
        }
        if ($row->getSourceProperty('link_path') == 'node/add/want') {
          $row->setSourceProperty('link_path', 'ad/add/want');
        }
    }

    if ($migration->id()== 'd7_view_mode' and $row->getSourceProperty('bundle') == 'proposition') {
      throw new MigrateSkipRowException();
    }
  }

  /**
   * @param MigratePostRowSaveEvent $event
   */
  function migratePostRowSave(MigratePostRowSaveEvent $event) {
    if ($event->getMigration()->getBaseId() == 'd7_smallad') {
      // Hack. I don't know what this field does or why it imports as 0 when it should be one.
      // This will probably break on multilingual sites.
      \Drupal::database()->update('smallad_field_data')->fields(['default_langcode' => 1])->execute();
      $event->logMessage('default_langcode set to 1 for all smallads', 'warning');
    }
    if ($event->getMigration()->id() == 'd7_smallad:offer') {
      if ($img_items = $event->getRow()->getDestinationProperty('image')) {
        foreach ($img_items as $value) {
          if ($file = File::load($value['target_id'])) {
            $file_uri = $file->getFileUri();
            if (file_exists($file_uri)) {
              $dest = str_replace('public://', 'public://smallads/', $file_uri);
              \Drupal::service('file_system')->move($file_uri, $dest);
              $file->setFileUri($dest);
              $file->save();
            }
          }
        }
      }
    }
  }

}
