<?php

declare(strict_types=1);

namespace Drupal\Tests\smart_db_tools\Kernel\mysql\Console;

use Drupal\mysql\Driver\Database\mysql\Connection as MysqlConnection;
use Drupal\smart_db_tools\Command\SmartDbDumpCommand;
use Symfony\Component\Console\Tester\CommandTester;

// cspell:ignore sdbt

/**
 * Test that the SmartDbDumpCommand with split destination works correctly.
 *
 * @group smart_db_tools
 * @group console
 *
 * @see \Drupal\Tests\system\Kernel\Scripts\DbDumpCommandTest
 */
class SmartDbDumpCommandToFilesTest extends SmartDbDumpCommandTest {

  /**
   * {@inheritdoc}
   */
  public function testDbDumpCommand($options = NULL): void {
    $options = $options ?? [
      '--split-destination' => implode(DIRECTORY_SEPARATOR, [
        $this->siteDirectory,
        // Use a random subdirectory.
        $this->randomMachineName(),
        $this->randomMachineName() . '.php',
      ]),
    ];
    $main_file_path = $options['--split-destination'];
    $main_file_name = basename($main_file_path, '.php');
    $directory = substr($main_file_path, 0, -4);
    $command = new SmartDbDumpCommand();
    $command_tester = new CommandTester($command);
    $command_tester->execute($options);

    // Assert that the main file only contains includes.
    $main_file_content = file_get_contents($main_file_path);
    $this->assertStringContainsString("include '$main_file_name' . DIRECTORY_SEPARATOR . 'router.php';", $main_file_content);
    $this->assertStringNotContainsString("createTable('router", $main_file_content, 'Table router found');
    $this->assertStringNotContainsString("insert('router", $main_file_content, 'Insert found');
    $this->assertStringNotContainsString("'name' => 'test", $main_file_content, 'Insert name field found');
    $this->assertStringNotContainsString("'path' => 'test", $main_file_content, 'Insert path field found');
    $this->assertStringNotContainsString("'pattern_outline' => 'test", $main_file_content, 'Insert pattern_outline field found');
    $version = \Drupal::VERSION;
    $version_normalized = preg_replace(
      [
        '/\.x-dev$/',
        '/\.(\d+)-dev$/',
      ],
      [
        '.0',
        '.$1',
      ],
      $version
    );
    if (version_compare($version_normalized, '9.2', 'ge')) {
      $this->assertStringContainsString("// phpcs:ignoreFile", $main_file_content);
    }
    else {
      $this->assertStringContainsString("// @codingStandardsIgnoreFile", $main_file_content);
    }
    $this->assertStringContainsString("This file was generated by the Drupal {$version} db-tools.php script.", $main_file_content);

    // Assert that table file exists, and it contains the expected commands.
    $router_content = file_get_contents($directory . DIRECTORY_SEPARATOR . 'router.php');
    $this->assertStringNotContainsString("include '$main_file_name' . DIRECTORY_SEPARATOR", $router_content);
    $this->assertStringContainsString("createTable('router", $router_content, 'Table router found');
    $this->assertStringContainsString("insert('router", $router_content, 'Insert found');
    $this->assertStringContainsString("'name' => 'test", $router_content, 'Insert name field found');
    $this->assertStringContainsString("'path' => 'test", $router_content, 'Insert path field found');
    $this->assertStringContainsString("'pattern_outline' => 'test", $router_content, 'Insert pattern_outline field found');
    if (version_compare($version_normalized, '9.2', 'ge')) {
      $this->assertStringContainsString("// phpcs:ignoreFile", $router_content);
    }
    else {
      $this->assertStringContainsString("// @codingStandardsIgnoreFile", $router_content);
    }
    $this->assertStringContainsString("This file was generated by the Drupal {$version} db-tools.php script.", $router_content);
  }

  /**
   * {@inheritdoc}
   */
  public function testSchemaOnly(): void {
    $main_file_name = $this->randomMachineName();
    $directory = implode(DIRECTORY_SEPARATOR, [
      $this->siteDirectory,
      // Use a random subdirectory.
      $this->randomMachineName(),
      $main_file_name,
    ]);
    $main_file_path = $directory . '.php';
    $command = new SmartDbDumpCommand();
    $command_tester = new CommandTester($command);
    $command_tester->execute([
      '--split-destination' => $main_file_path,
      '--schema-only' => 'router',
    ]);

    // Assert that insert statement doesn't exist for schema only table.
    $router_content = file_get_contents($directory . DIRECTORY_SEPARATOR . 'router.php');
    $this->assertStringContainsString("createTable('router", $router_content, 'Table router found');
    $this->assertStringNotContainsString("insert('router", $router_content, 'Insert not found');
    $this->assertStringNotContainsString("'name' => 'test", $router_content, 'Insert name field not found');
    $this->assertStringNotContainsString("'path' => 'test", $router_content, 'Insert path field not found');
    $this->assertStringNotContainsString("'pattern_outline' => 'test", $router_content, 'Insert pattern_outline field not found');

    // Assert that insert statement doesn't exist for wildcard schema only
    // match.
    $command_tester->execute([
      '--split-destination' => $main_file_path,
      '--schema-only' => 'route.*',
    ]);
    $router_content = file_get_contents($directory . DIRECTORY_SEPARATOR . 'router.php');
    $this->assertStringContainsString("createTable('router", $router_content, 'Table router found');
    $this->assertStringNotContainsString("insert('router", $router_content, 'Insert not found');
    $this->assertStringNotContainsString("'name' => 'test", $router_content, 'Insert name field not found');
    $this->assertStringNotContainsString("'path' => 'test", $router_content, 'Insert path field not found');
    $this->assertStringNotContainsString("'pattern_outline' => 'test", $router_content, 'Insert pattern_outline field not found');
  }

  /**
   * {@inheritdoc}
   */
  public function testDbDumpWithSplit(): void {
    $this->addSplitTestTable();

    $main_file_name = $this->randomMachineName();
    $main_subdirectory = implode(DIRECTORY_SEPARATOR, [
      $this->siteDirectory,
      // Use a random subdirectory.
      $this->randomMachineName(),
      $main_file_name,
    ]);
    $main_file_path = $main_subdirectory . '.php';
    $this->testDbDumpCommand([
      '--split-destination' => $main_file_path,
      '--subsplit-limit' => 100,
      '--insert-count' => 10,
    ]);

    // Assert that the main file includes 'sdbt_split_test'.
    $main_file_content = file_get_contents($main_file_path);
    $this->assertStringContainsString("include '$main_file_name' . DIRECTORY_SEPARATOR . 'sdbt_split_test.php';", $main_file_content);

    // Assert that the main subsplit table file of 'sdbt_split_test' table
    // exists, and it also only contains imports.
    $sdbt_split_test_main_content = file_get_contents($main_subdirectory . DIRECTORY_SEPARATOR . 'sdbt_split_test.php');
    $this->assertStringContainsString("include 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_0.php';", $sdbt_split_test_main_content);
    $this->assertStringContainsString("include 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_1.php';", $sdbt_split_test_main_content);
    $this->assertStringContainsString("include 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_2.php';", $sdbt_split_test_main_content);
    $this->assertStringNotContainsString("include 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_3.php';", $sdbt_split_test_main_content);
    $this->assertStringNotContainsString("createTable('sdbt_split_test", $sdbt_split_test_main_content, 'Table sdbt_split_test_0 found');
    $this->assertStringNotContainsString("insert('sdbt_split_test", $sdbt_split_test_main_content, 'Insert found');
    $this->assertStringNotContainsString("'id' => '1'", $sdbt_split_test_main_content, 'Insert id "1" field found');
    $this->assertStringNotContainsString("'data' => 'i:1;'", $sdbt_split_test_main_content, 'Insert data "1" field found');

    // Assert that table subsplit files exist and contain the expected commands.
    // Subsplit #0.
    $sdbt_split_test_0_content = file_get_contents($main_subdirectory . DIRECTORY_SEPARATOR . 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_0.php');
    $this->assertStringNotContainsString("include '", $sdbt_split_test_0_content);
    $this->assertStringContainsString("createTable('sdbt_split_test", $sdbt_split_test_0_content, 'Table sdbt_split_test found');
    $this->assertStringContainsString("insert('sdbt_split_test", $sdbt_split_test_0_content, 'Insert found');
    $this->assertStringContainsString("'id' => '1'", $sdbt_split_test_0_content, 'Insert id "1" field found');
    $this->assertStringContainsString("'data' => 'i:1;'", $sdbt_split_test_0_content, 'Insert data "1" field found');
    $this->assertStringContainsString("'id' => '100'", $sdbt_split_test_0_content, 'Insert id "100" field found');
    $this->assertStringContainsString("'data' => 'i:100;'", $sdbt_split_test_0_content, 'Insert data "100" field found');
    $this->assertStringNotContainsString("'id' => '101'", $sdbt_split_test_0_content, 'Insert id "101" field found');
    $this->assertStringNotContainsString("'data' => 'i:101;'", $sdbt_split_test_0_content, 'Insert data "101" field found');

    // Subsplit #1.
    $sdbt_split_test_1_content = file_get_contents($main_subdirectory . DIRECTORY_SEPARATOR . 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_1.php');
    $this->assertStringNotContainsString("include '", $sdbt_split_test_1_content);
    $this->assertStringNotContainsString("createTable('sdbt_split_test", $sdbt_split_test_1_content, 'Table creation sdbt_split_test found');
    $this->assertStringContainsString("insert('sdbt_split_test", $sdbt_split_test_1_content, 'Insert not found');
    $this->assertStringNotContainsString("'id' => '1'", $sdbt_split_test_1_content, 'Insert id "1" field found');
    $this->assertStringNotContainsString("'data' => 'i:1;'", $sdbt_split_test_1_content, 'Insert data "1" field found');
    $this->assertStringContainsString("'id' => '101'", $sdbt_split_test_1_content, 'Insert id "101" field not found');
    $this->assertStringContainsString("'data' => 'i:101;'", $sdbt_split_test_1_content, 'Insert data "101" field not found');
    $this->assertStringContainsString("'id' => '200'", $sdbt_split_test_1_content, 'Insert id "200" field not found');
    $this->assertStringContainsString("'data' => 'i:200;'", $sdbt_split_test_1_content, 'Insert data "200" field not found');
    $this->assertStringNotContainsString("'id' => '201'", $sdbt_split_test_1_content, 'Insert id "201" field found');
    $this->assertStringNotContainsString("'data' => 'i:201;'", $sdbt_split_test_1_content, 'Insert data "201" field found');

    // Subsplit #2.
    $sdbt_split_test_2_content = file_get_contents($main_subdirectory . DIRECTORY_SEPARATOR . 'sdbt_split_test' . DIRECTORY_SEPARATOR . 'sdbt_split_test_2.php');
    $this->assertStringNotContainsString("include '", $sdbt_split_test_2_content);
    $this->assertStringNotContainsString("createTable('sdbt_split_test", $sdbt_split_test_2_content, 'Table creation sdbt_split_test found');
    $this->assertStringContainsString("insert('sdbt_split_test", $sdbt_split_test_2_content, 'Insert not found');
    $this->assertStringNotContainsString("'id' => '1'", $sdbt_split_test_2_content, 'Insert id "1" field found');
    $this->assertStringNotContainsString("'data' => 'i:1;'", $sdbt_split_test_2_content, 'Insert data "1" field found');
    $this->assertStringNotContainsString("'id' => '200'", $sdbt_split_test_2_content, 'Insert id "200" field found');
    $this->assertStringNotContainsString("'data' => 'i:200;'", $sdbt_split_test_2_content, 'Insert data "200" field found');
    $this->assertStringContainsString("'id' => '201'", $sdbt_split_test_2_content, 'Insert id "201" field not found');
    $this->assertStringContainsString("'data' => 'i:201;'", $sdbt_split_test_2_content, 'Insert data "201" field not found');
    $this->assertStringNotContainsString("'id' => '200'", $sdbt_split_test_2_content, 'Insert id "200" field found');
    $this->assertStringNotContainsString("'data' => 'i:200;'", $sdbt_split_test_2_content, 'Insert data "200" field found');
  }

  /**
   * {@inheritdoc}
   */
  public function testDbDumpWithNoSplit(): void {
    $this->addSplitTestTable();

    $main_file_name = $this->randomMachineName();
    $main_subdirectory = implode(DIRECTORY_SEPARATOR, [
      $this->siteDirectory,
      // Use a random subdirectory.
      $this->randomMachineName(),
      $main_file_name,
    ]);
    $main_file_path = $main_subdirectory . '.php';
    $this->testDbDumpCommand([
      '--split-destination' => $main_file_path,
      '--subsplit-limit' => 0,
    ]);

    // Assert that the main file includes 'sdbt_split_test'.
    $main_file_content = file_get_contents($main_file_path);
    $this->assertStringContainsString("include '$main_file_name' . DIRECTORY_SEPARATOR . 'sdbt_split_test.php';", $main_file_content);

    // Assert that the main subsplit table file of 'sdbt_split_test' table
    // exists, and it also only contains imports.
    $sdbt_split_test_content = file_get_contents($main_subdirectory . DIRECTORY_SEPARATOR . 'sdbt_split_test.php');
    $this->assertStringNotContainsString("include '", $sdbt_split_test_content);
    $this->assertStringContainsString("createTable('sdbt_split_test", $sdbt_split_test_content, 'Table sdbt_split_test found');
    $this->assertStringContainsString("insert('sdbt_split_test", $sdbt_split_test_content, 'Insert found');
    $this->assertStringContainsString("'id' => '1'", $sdbt_split_test_content, 'Insert id "1" field found');
    $this->assertStringContainsString("'data' => 'i:1;'", $sdbt_split_test_content, 'Insert data "1" field found');
    $this->assertStringContainsString("'id' => '100'", $sdbt_split_test_content, 'Insert id "100" field found');
    $this->assertStringContainsString("'data' => 'i:100;'", $sdbt_split_test_content, 'Insert data "100" field found');
    $this->assertStringContainsString("'id' => '101'", $sdbt_split_test_content, 'Insert id "101" field not found');
    $this->assertStringContainsString("'data' => 'i:101;'", $sdbt_split_test_content, 'Insert data "101" field not found');
    $this->assertStringContainsString("'id' => '200'", $sdbt_split_test_content, 'Insert id "200" field not found');
    $this->assertStringContainsString("'data' => 'i:200;'", $sdbt_split_test_content, 'Insert data "200" field not found');
    $this->assertStringContainsString("'id' => '201'", $sdbt_split_test_content, 'Insert id "201" field not found');
    $this->assertStringContainsString("'data' => 'i:201;'", $sdbt_split_test_content, 'Insert data "201" field not found');
  }

  /**
   * Adds a 'sdbt_split_test' table and inserts the given number of records.
   *
   * @param int $items
   *   How many records to add.
   * @param int $big_items
   *   How many items to add with big (1M) data.
   *
   * @throws \Exception
   */
  protected function addSplitTestTable(int $items = 202, int $big_items = 3): void {
    $connection = $this->container->get('database');
    $this->assertInstanceOf(MysqlConnection::class, $connection);
    $connection->schema()->createTable('sdbt_split_test', [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
          'size' => 'normal',
          'unsigned' => TRUE,
        ],
        'data' => [
          'type' => 'blob',
          'not null' => FALSE,
          'size' => 'big',
        ],
      ],
      'primary key' => ['id'],
      'indexes' => [],
      'mysql_character_set' => 'utf8',
    ]);

    $id = 1;
    $big_item_nth = $big_items > 0
      ? ceil($items / $big_items)
      : $items + 1;
    while ($id < $items) {
      $data = ($id % $big_item_nth) === 0
        ? str_repeat('s', 1024 * 1024)
        : $id;
      $connection->insert('sdbt_split_test')
        ->fields([
          'data' => serialize($data),
        ])
        ->execute();
      $id++;
    }
  }

}
