<?php

declare(strict_types=1);

namespace Drupal\soap_manager\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\soap_manager\Entity\SoapManagerLogRequestInterface;
use Drupal\soap_manager\Entity\SoapManagerLogResponseInterface;

/**
 * Controller for SOAP logs overview.
 */
class SoapLogController extends ControllerBase {

  /**
   * Displays an overview of SOAP logs.
   *
   * @return array
   *   A render array.
   */
  public function overview(): array {
    $build = [];

    $build['intro'] = [
      '#markup' => '<p>' . $this->t('View and manage SOAP request and response logs.') . '</p>',
    ];

    $build['links'] = [
      '#theme' => 'item_list',
      '#items' => [
        Link::createFromRoute(
          $this->t('View Request Logs'),
          'entity.soap_log_request.collection'
        ),
        Link::createFromRoute(
          $this->t('View Response Logs'),
          'entity.soap_log_response.collection'
        ),
      ],
    ];

    return $build;
  }

  /**
   * Title callback for request log entity.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerLogRequestInterface $soap_log_request
   *   The request log entity.
   *
   * @return string
   *   The title.
   */
  public function requestTitle(SoapManagerLogRequestInterface $soap_log_request): TranslatableMarkup {
    return $this->t('SOAP Request Log #@id', ['@id' => $soap_log_request->id()]);
  }

  /**
   * Title callback for response log entity.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerLogResponseInterface $soap_log_response
   *   The response log entity.
   *
   * @return string
   *   The title.
   */
  public function responseTitle(SoapManagerLogResponseInterface $soap_log_response): TranslatableMarkup {
    return $this->t('SOAP Response Log #@id', ['@id' => $soap_log_response->id()]);
  }

}


