<?php

declare(strict_types=1);

namespace Drupal\soap_manager\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\file\FileInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for serving SOAP log XML files.
 */
class SoapLogFileController extends ControllerBase {

  /**
   * Custom access check for downloading SOAP log files.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  public function access(FileInterface $file, AccountInterface $account): AccessResult {
    // Check if user has permission to view SOAP logs.
    if (!$account->hasPermission('view soap logs')) {
      return AccessResult::forbidden('User does not have permission to view SOAP logs.');
    }

    // Verify the file is in the SOAP logs directory.
    $uri = $file->getFileUri();
    if (!str_starts_with($uri, 'private://soap_logs/')) {
      return AccessResult::forbidden('File is not a SOAP log file.');
    }

    // Check if the file exists.
    $real_path = \Drupal::service('file_system')->realpath($uri);
    if (!$real_path || !file_exists($real_path)) {
      return AccessResult::forbidden('File does not exist.');
    }

    return AccessResult::allowed();
  }

  /**
   * Serves a SOAP log XML file.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
   *   The file response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If the file is not found.
   */
  public function download(FileInterface $file): BinaryFileResponse {
    $uri = $file->getFileUri();
    $path = \Drupal::service('file_system')->realpath($uri);

    if (!$path || !file_exists($path)) {
      throw new NotFoundHttpException();
    }

    $response = new BinaryFileResponse($path);
    $response->headers->set('Content-Type', 'application/xml');
    $response->headers->set('Content-Disposition', 'inline; filename="' . $file->getFilename() . '"');

    return $response;
  }

}

