<?php

namespace Drupal\soap_manager\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\soap_manager\Entity\SoapManagerInterface;
use Drupal\soap_manager\Service\SoapServer;
use Drupal\soap_manager\WsdlGenerator;
use Drupal\soap_manager\Service\SoapSecurity;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for handling SOAP endpoint requests.
 */
class SoapManagerController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The SOAP server service.
   *
   * @var \Drupal\soap_manager\Service\SoapServer
   */
  protected $soapServer;

  /**
   * The WSDL generator service.
   *
   * @var \Drupal\soap_manager\WsdlGenerator
   */
  protected $wsdlGenerator;

  /**
   * The SOAP security service.
   *
   * @var \Drupal\soap_manager\Service\SoapSecurity
   */
  protected $soapSecurity;

  /**
   * Constructs a new SoapManagerController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\soap_manager\Service\SoapServer $soap_server
   *   The SOAP server service.
   * @param \Drupal\soap_manager\WsdlGenerator $wsdl_generator
   *   The WSDL generator service.
   * @param \Drupal\soap_manager\Service\SoapSecurity $soap_security
   *   The SOAP security service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    SoapServer $soap_server,
    WsdlGenerator $wsdl_generator,
    SoapSecurity $soap_security
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->soapServer = $soap_server;
    $this->wsdlGenerator = $wsdl_generator;
    $this->soapSecurity = $soap_security;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('soap_manager.server'),
      $container->get('soap_manager.wsdl_generator'),
      $container->get('soap_manager.security')
    );
  }

  /**
   * Handles a request to a SOAP endpoint.
   *
   * This method handles both GET requests for WSDL documents and POST requests
   * for SOAP operations.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response.
   */
  public function handleRequest(Request $request, SoapManagerInterface $soap_manager) {
    // Check if the endpoint is enabled.
    if (!$soap_manager->status()) {
      return new Response(
        'This SOAP endpoint is currently disabled.',
        Response::HTTP_SERVICE_UNAVAILABLE
      );
    }

    // Check if authentication is required and validate the request.
    if ($soap_manager->isAuthRequired() && !$this->soapSecurity->validateAuth($request, TRUE)) {
      return new Response(
        'Authentication required to access this SOAP endpoint.',
        Response::HTTP_UNAUTHORIZED,
        ['WWW-Authenticate' => 'Basic realm="SOAP Endpoint"']
      );
    }

    // If this is a GET request with ?wsdl parameter, return the WSDL document.
    if ($request->getMethod() === 'GET' && $request->query->has('wsdl')) {
      return $this->getWsdl($request, $soap_manager);
    }

    // If this is a POST request, handle the SOAP operation.
    if ($request->getMethod() === 'POST') {
      return $this->handleSoapOperation($request, $soap_manager);
    }

    // If we get here, it's neither a WSDL request nor a SOAP operation.
    // Return a simple info page about the SOAP endpoint if user authenticated.
    if (\Drupal::currentUser()->isAuthenticated()) {
      return $this->endpointInfo($soap_manager);
    }
    else {
      return new Response(
        'Authentication required to access this SOAP endpoint.',
        Response::HTTP_UNAUTHORIZED,
      );
    }
  }

  /**
   * Generates and returns the WSDL document for a SOAP endpoint.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response containing the WSDL document.
   */
  protected function getWsdl(Request $request, SoapManagerInterface $soap_manager) {
    // Check if a custom WSDL path is specified.
    $wsdl_path = $soap_manager->getWsdlPath();
    if (!empty($wsdl_path)) {
      // Use the custom WSDL file.
      $wsdl_content = $this->loadCustomWsdl($wsdl_path);
      if ($wsdl_content !== FALSE) {
        // Create and return the response with the custom WSDL.
        $response = new Response($wsdl_content);
        $response->headers->set('Content-Type', 'application/xml');
        return $response;
      }
      // If loading the custom WSDL failed, log a warning and fall back to generated WSDL.
      \Drupal::logger('soap_manager')->warning('Failed to load custom WSDL file from %path for endpoint %endpoint. Falling back to generated WSDL.', [
        '%path' => $wsdl_path,
        '%endpoint' => $soap_manager->label(),
      ]);
    }

    // Generate the base URL for WSDL.
    $base_url = $request->getSchemeAndHttpHost();

    // Generate the WSDL document.
    $wsdl = $this->wsdlGenerator->generateWsdl($soap_manager, $base_url);

    // Create and return the response.
    $response = new Response($wsdl);
    $response->headers->set('Content-Type', 'application/xml');
    return $response;
  }

  /**
   * Loads a custom WSDL file from the specified path.
   *
   * @param string $wsdl_path
   *   The path to the WSDL file, either absolute or relative to the Drupal root.
   *
   * @return string|false
   *   The contents of the WSDL file, or FALSE if the file cannot be read.
   */
  protected function loadCustomWsdl($wsdl_path) {
    // Check if the path is absolute.
    if (!file_exists($wsdl_path)) {
      // Try relative to Drupal root.
      $drupal_root = \Drupal::root();
      $wsdl_path = $drupal_root . '/' . ltrim($wsdl_path, '/');

      if (!file_exists($wsdl_path)) {
        return FALSE;
      }
    }

    // Load the file contents.
    $wsdl_content = file_get_contents($wsdl_path);
    if ($wsdl_content === FALSE) {
      return FALSE;
    }

    return $wsdl_content;
  }

  /**
   * Handles a SOAP operation.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response.
   */
  protected function handleSoapOperation(Request $request, SoapManagerInterface $soap_manager) {
    // Get the raw POST data.
    $soap_request = $request->getContent();

    // If the request is empty, return an error.
    if (empty($soap_request)) {
      return new Response(
        'Empty SOAP request.',
        Response::HTTP_BAD_REQUEST,
        ['Content-Type' => 'text/plain']
      );
    }

    // Validate XML structure before processing.
    libxml_use_internal_errors(true);
    $doc = simplexml_load_string($soap_request);
    if ($doc === false) {
      $errors = libxml_get_errors();
      libxml_clear_errors();

      $error_message = 'Invalid XML in SOAP request: ';
      foreach ($errors as $error) {
        $error_message .= $error->message . ' ';
      }

      return new Response(
        $error_message,
        Response::HTTP_BAD_REQUEST,
        ['Content-Type' => 'text/plain']
      );
    }

    // Generate the base URL for WSDL.
    $base_url = $request->getSchemeAndHttpHost();
    $wsdl_url = $base_url . Url::fromRoute('entity.soap_manager.wsdl', [
      'soap_manager' => $soap_manager->id(),
    ])->toString();

    // If a custom WSDL path is specified, use that instead.
    $wsdl_path = $soap_manager->getWsdlPath();
    if (!empty($wsdl_path)) {
      // Check if the file exists
      $wsdl_file_path = $wsdl_path;
      if (!file_exists($wsdl_file_path)) {
        // Try relative to Drupal root
        $drupal_root = \Drupal::root();
        $wsdl_file_path = $drupal_root . '/' . ltrim($wsdl_path, '/');
      }

      if (file_exists($wsdl_file_path)) {
        $wsdl_url = $wsdl_file_path;
      } else {
        // Log a warning but continue with the generated WSDL URL
        \Drupal::logger('soap_manager')->warning('Could not find custom WSDL file at %path for SOAP operation. Using generated WSDL instead.', [
          '%path' => $wsdl_path,
        ]);
      }
    }

    $options = [
      'uri' => $base_url . '/' . $soap_manager->getPath(),
      'encoding' => 'UTF-8',
      'cache_wsdl' => WSDL_CACHE_NONE,
      'features' => SOAP_SINGLE_ELEMENT_ARRAYS
    ];

    // Handle the SOAP request.
    $soap_response = $this->soapServer->handleRequest(
      $soap_manager->id(),
      $wsdl_url,
      $options,
      $soap_request
    );

    // Create a new Response object with the SOAP response
    $response = new Response($soap_response);

    // Ensure Content-Type is set properly, but only once
    // We completely reset the headers to avoid duplicates
    $response->headers->set('Content-Type', 'text/xml; charset=utf-8');

    // Add cache-control headers to prevent caching of SOAP responses
    $response->headers->set('Cache-Control', 'no-cache, no-store, must-revalidate');
    $response->headers->set('Pragma', 'no-cache');
    $response->headers->set('Expires', '0');

    return $response;
  }

  /**
   * Displays information about the SOAP endpoint.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return array
   *   A render array for the endpoint info page.
   */
  protected function endpointInfo(SoapManagerInterface $soap_manager) {
    $build = [
      '#theme' => 'soap_manager',
      '#soap_manager' => $soap_manager,
      '#wsdl_info' => [
        'is_custom' => !empty($soap_manager->getWsdlPath()),
        'wsdl_path' => $soap_manager->getWsdlPath(),
      ],
    ];

    return $build;
  }

  /**
   * View handler for an endpoint's admin page.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return array
   *   A render array.
   */
  public function viewEndpoint(SoapManagerInterface $soap_manager) {
    return $this->endpointInfo($soap_manager);
  }

  /**
   * Title callback for endpoint pages.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return string
   *   The endpoint title.
   */
  public function getEndpointTitle(SoapManagerInterface $soap_manager) {
    return $this->t('SOAP Endpoint: @label', ['@label' => $soap_manager->label()]);
  }

  /**
   * Generates and returns the WSDL document for a SOAP endpoint in the admin UI.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager
   *   The SOAP endpoint entity.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response containing the WSDL document.
   */
  public function getWsdlDocument(SoapManagerInterface $soap_manager) {
    // Check if a custom WSDL path is specified.
    $wsdl_path = $soap_manager->getWsdlPath();
    if (!empty($wsdl_path)) {
      // Use the custom WSDL file.
      $wsdl_content = $this->loadCustomWsdl($wsdl_path);
      if ($wsdl_content !== FALSE) {
        // Create and return the response with the custom WSDL.
        $response = new Response($wsdl_content);
        $response->headers->set('Content-Type', 'application/xml');
        return $response;
      }
      // If loading custom WSDL failed, log a warning and fall back to generated WSDL.
      \Drupal::logger('soap_manager')->warning('Failed to load custom WSDL file from %path for endpoint %endpoint in admin UI. Falling back to generated WSDL.', [
        '%path' => $wsdl_path,
        '%endpoint' => $soap_manager->label(),
      ]);
    }

    // Create a mock request to get the scheme and HTTP host
    $request = \Drupal::request();
    $base_url = $request->getSchemeAndHttpHost();

    // Generate the WSDL document
    $wsdl = $this->wsdlGenerator->generateWsdl($soap_manager, $base_url);

    // Create and return the response
    $response = new Response($wsdl);
    $response->headers->set('Content-Type', 'application/xml');
    return $response;
  }

}
