<?php

namespace Drupal\soap_manager\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the SOAP Endpoint entity.
 *
 * @ConfigEntityType(
 *   id = "soap_manager",
 *   label = @Translation("SOAP Endpoint"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\soap_manager\SoapManagerListBuilder",
 *     "form" = {
 *       "add" = "Drupal\soap_manager\Form\SoapManagerForm",
 *       "edit" = "Drupal\soap_manager\Form\SoapManagerForm",
 *       "delete" = "Drupal\soap_manager\Form\SoapManagerDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "soap_manager",
 *   admin_permission = "administer soap endpoints",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "status" = "status"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "path",
 *     "description",
 *     "status",
 *     "resources",
 *     "auth_required",
 *     "auth_settings",
 *     "wsdl_path"
 *   },
 *   links = {
 *     "add-form" = "/admin/structure/soap-manager/add",
 *     "edit-form" = "/admin/structure/soap-manager/{soap_manager}/edit",
 *     "delete-form" = "/admin/structure/soap-manager/{soap_manager}/delete",
 *     "collection" = "/admin/structure/soap-manager"
 *   }
 * )
 */
class SoapManager extends ConfigEntityBase implements SoapManagerInterface {

  /**
   * The SOAP endpoint ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The SOAP endpoint label.
   *
   * @var string
   */
  protected $label;

  /**
   * The path for the SOAP endpoint.
   *
   * @var string
   */
  protected $path;

  /**
   * The description of the SOAP endpoint.
   *
   * @var string
   */
  protected $description = '';

  /**
   * Enabled SOAP resource plugin IDs.
   *
   * @var array
   */
  protected $resources = [];

  /**
   * Whether authentication is required for this endpoint.
   *
   * @var bool
   */
  protected $auth_required = FALSE;

  /**
   * Authentication settings for this endpoint.
   *
   * @var array
   */
  protected $auth_settings = [];

  /**
   * Path to custom WSDL file.
   *
   * @var string
   */
  protected $wsdl_path = '';

  /**
   * {@inheritdoc}
   */
  public function getPath() {
    return $this->path;
  }

  /**
   * {@inheritdoc}
   */
  public function setPath($path) {
    $this->path = $path;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->description = $description;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getEnabledResources() {
    return $this->resources;
  }

  /**
   * {@inheritdoc}
   */
  public function setEnabledResources(array $resources) {
    $this->resources = $resources;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasResource($resource_id) {
    return array_key_exists($resource_id, $this->resources);
  }

  /**
   * Get resource soap method.
   *
   * @param string $plugin_id
   *   Plugin id.
   *
   * @return string|null
   *   Soap method.
   */
  public function getResourceSoapMethod(string $plugin_id): ?string {
    return $this->resources[$plugin_id] ?? NULL;
  }

  /**
   * Get resource id by soap method.
   *
   * @param string $method_to_find
   *   Method to find.
   *
   * @return string|null
   *   Resource id.
   */
  public function getResourceIdBySoapMethod(string $method_to_find): ?string {
    foreach ($this->resources as $plugin_id => $soap_method) {
      if ($method_to_find === $soap_method) {
        return $plugin_id;
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isAuthRequired() {
    return $this->auth_required;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthRequired($authRequired) {
    $this->auth_required = $authRequired;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthSettings() {
    return $this->auth_settings;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthSettings(array $settings) {
    $this->auth_settings = $settings;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getWsdlPath() {
    return $this->wsdl_path;
  }

  /**
   * {@inheritdoc}
   */
  public function setWsdlPath($wsdl_path) {
    $this->wsdl_path = $wsdl_path;
    return $this;
  }

}
