<?php

namespace Drupal\soap_manager\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Provides an interface defining a SOAP endpoint entity.
 */
interface SoapManagerInterface extends ConfigEntityInterface {

  /**
   * Returns the path for this SOAP endpoint.
   *
   * @return string
   *   The path for this SOAP endpoint.
   */
  public function getPath();

  /**
   * Sets the path for this SOAP endpoint.
   *
   * @param string $path
   *   The path for this SOAP endpoint.
   *
   * @return $this
   */
  public function setPath($path);

  /**
   * Returns the description for this SOAP endpoint.
   *
   * @return string
   *   The description for this SOAP endpoint.
   */
  public function getDescription();

  /**
   * Sets the description for this SOAP endpoint.
   *
   * @param string $description
   *   The description for this SOAP endpoint.
   *
   * @return $this
   */
  public function setDescription($description);

  /**
   * Returns the enabled SOAP resources for this endpoint.
   *
   * @return array
   *   An array of enabled resource plugin IDs.
   */
  public function getEnabledResources();

  /**
   * Sets the enabled SOAP resources for this endpoint.
   *
   * @param array $resources
   *   An array of resource plugin IDs to enable.
   *
   * @return $this
   */
  public function setEnabledResources(array $resources);

  /**
   * Returns whether the endpoint has a specific resource.
   *
   * @param string $resource_id
   *   The ID of the resource to check.
   *
   * @return bool
   *   TRUE if the resource exists, FALSE otherwise.
   */
  public function hasResource($resource_id);

  /**
   * Get resource soap method.
   *
   * @param string $plugin_id
   *   Plugin id.
   *
   * @return string|null
   *   Soap method.
   */
  public function getResourceSoapMethod(string $plugin_id): ?string;

  /**
   * Get resource id by soap method.
   *
   * @param string $method_to_find
   *   Method to find.
   *
   * @return string|null
   *   Resource id.
   */
  public function getResourceIdBySoapMethod(string $method_to_find): ?string;

  /**
   * Returns whether authentication is required for this endpoint.
   *
   * @return bool
   *   TRUE if authentication is required, FALSE otherwise.
   */
  public function isAuthRequired();

  /**
   * Sets whether authentication is required for this endpoint.
   *
   * @param bool $authRequired
   *   Whether authentication is required.
   *
   * @return $this
   */
  public function setAuthRequired($authRequired);

  /**
   * Returns the authentication settings for this endpoint.
   *
   * @return array
   *   The authentication settings.
   */
  public function getAuthSettings();

  /**
   * Sets the authentication settings for this endpoint.
   *
   * @param array $settings
   *   The authentication settings.
   *
   * @return $this
   */
  public function setAuthSettings(array $settings);

  /**
   * Returns the custom WSDL file path for this endpoint.
   *
   * @return string
   *   The path to the custom WSDL file, or an empty string if using generated WSDL.
   */
  public function getWsdlPath();

  /**
   * Sets the custom WSDL file path for this endpoint.
   *
   * @param string $wsdl_path
   *   The path to the custom WSDL file, or an empty string to use generated WSDL.
   *
   * @return $this
   */
  public function setWsdlPath($wsdl_path);

}
