<?php

declare(strict_types=1);

namespace Drupal\soap_manager\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the SOAP request log entity.
 *
 * @ContentEntityType(
 *   id = "soap_log_request",
 *   label = @Translation("SOAP Request Log"),
 *   label_collection = @Translation("SOAP Request Logs"),
 *   label_singular = @Translation("SOAP request log"),
 *   label_plural = @Translation("SOAP request logs"),
 *   label_count = @PluralTranslation(
 *     singular = "@count SOAP request log",
 *     plural = "@count SOAP request logs",
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\soap_manager\SoapManagerLogRequestListBuilder",
 *     "access" = "Drupal\soap_manager\SoapManagerLogAccessControlHandler",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "soap_log_request",
 *   admin_permission = "view soap logs",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   links = {
 *     "canonical" = "/admin/reports/soap-logs/request/{soap_log_request}",
 *     "delete-form" = "/admin/reports/soap-logs/request/{soap_log_request}/delete",
 *     "collection" = "/admin/reports/soap-logs/requests",
 *   },
 * )
 */
class SoapManagerLogRequest extends ContentEntityBase implements SoapManagerLogRequestInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getEndpointId(): ?string {
    return $this->get('endpoint_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setEndpointId(string $endpoint_id): self {
    $this->set('endpoint_id', $endpoint_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientIp(): ?string {
    return $this->get('client_ip')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setClientIp(string $ip): self {
    $this->set('client_ip', $ip);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    return (int) $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $timestamp): self {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getResponseLogId(): ?int {
    $responses = $this->entityTypeManager()->getStorage('soap_log_response')->loadByProperties([
      'request_log_id' => $this->id(),
    ]);
    $response = reset($responses);
    if (!$response instanceof SoapManagerLogResponseInterface) {
      return NULL;
    }

    return (int) $response->id() ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Owner field.
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['endpoint_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Endpoint ID'))
      ->setDescription(t('The SOAP endpoint ID.'))
      ->setSettings([
        'max_length' => 255,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['client_ip'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Client IP'))
      ->setDescription(t('The IP address of the client.'))
      ->setSettings([
        'max_length' => 45,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['request_xml'] = BaseFieldDefinition::create('file')
      ->setLabel(t('Request XML'))
      ->setDescription(t('The SOAP request XML file.'))
      ->setSettings([
        'uri_scheme' => 'private',
        'file_directory' => 'soap_logs/requests',
        'file_extensions' => 'xml',
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'file_default',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the log was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the log was last edited.'));

    return $fields;
  }

}


