<?php

declare(strict_types=1);

namespace Drupal\soap_manager\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the SOAP response log entity.
 *
 * @ContentEntityType(
 *   id = "soap_log_response",
 *   label = @Translation("SOAP Response Log"),
 *   label_collection = @Translation("SOAP Response Logs"),
 *   label_singular = @Translation("SOAP response log"),
 *   label_plural = @Translation("SOAP response logs"),
 *   label_count = @PluralTranslation(
 *     singular = "@count SOAP response log",
 *     plural = "@count SOAP response logs",
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\soap_manager\SoapManagerLogResponseListBuilder",
 *     "access" = "Drupal\soap_manager\SoapManagerLogAccessControlHandler",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "soap_log_response",
 *   admin_permission = "view soap logs",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "canonical" = "/admin/reports/soap-logs/response/{soap_log_response}",
 *     "delete-form" = "/admin/reports/soap-logs/response/{soap_log_response}/delete",
 *     "collection" = "/admin/reports/soap-logs/responses",
 *   },
 * )
 */
class SoapManagerLogResponse extends ContentEntityBase implements SoapManagerLogResponseInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public function getRequestLogId(): ?int {
    return (int) $this->get('request_log_id')->target_id ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setRequestLogId(int $request_log_id): self {
    $this->set('request_log_id', $request_log_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusCode(): ?int {
    return (int) $this->get('status_code')->value ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setStatusCode(int $status_code): self {
    $this->set('status_code', $status_code);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isSuccess(): bool {
    return (bool) $this->get('success')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setSuccess(bool $success): self {
    $this->set('success', $success);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    return (int) $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $timestamp): self {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['request_log_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Request Log'))
      ->setDescription(t('The associated request log.'))
      ->setSetting('target_type', 'soap_log_request')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'entity_reference_label',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status_code'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('HTTP Status Code'))
      ->setDescription(t('The HTTP status code of the response.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_integer',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['success'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Success'))
      ->setDescription(t('Whether the response was successful.'))
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'boolean',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['response_xml'] = BaseFieldDefinition::create('file')
      ->setLabel(t('Response XML'))
      ->setDescription(t('The SOAP response XML file.'))
      ->setSettings([
        'uri_scheme' => 'private',
        'file_directory' => 'soap_logs/responses',
        'file_extensions' => 'xml',
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'file_default',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the log was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the log was last edited.'));

    return $fields;
  }

}


