<?php

declare(strict_types=1);

namespace Drupal\soap_manager\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure SOAP Endpoint settings for this site.
 */
class ModuleSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'soap_manager_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['soap_manager.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('soap_manager.settings');

    // General settings section
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General settings'),
      '#open' => TRUE,
    ];

    $form['general']['base_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base path for SOAP endpoints'),
      '#description' => $this->t('The base path prefix for all SOAP endpoints (e.g., "soap"). Do not include leading or trailing slashes.'),
      '#default_value' => $config->get('base_path') ?? 'soap',
    ];

    $form['general']['auth_methods'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Authentication methods'),
      '#description' => $this->t('Select which authentication methods are available for SOAP endpoints.'),
      '#options' => [
        'basic' => $this->t('HTTP Basic Authentication'),
        'drupal' => $this->t('Drupal authentication'),
        // Future: Add more auth methods
      ],
      '#default_value' => $config->get('auth_methods') ?? ['drupal'],
    ];

    // Logging settings section
    $form['logging'] = [
      '#type' => 'details',
      '#title' => $this->t('Logging Settings'),
      '#open' => TRUE,
    ];

    $form['logging']['log_requests'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log SOAP Requests'),
      '#description' => $this->t('Log all incoming SOAP requests.'),
      '#default_value' => $config->get('logging.log_requests') ?: TRUE,
    ];

    $form['logging']['log_responses'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log SOAP Responses'),
      '#description' => $this->t('Log all outgoing SOAP responses.'),
      '#default_value' => $config->get('logging.log_responses') ?: TRUE,
    ];

    $form['logging']['log_errors'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log SOAP Errors'),
      '#description' => $this->t('Log all SOAP errors.'),
      '#default_value' => $config->get('logging.log_errors') ?: TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Ensure base path doesn't have leading/trailing slashes.
    $base_path = $form_state->getValue('base_path');
    if (!empty($base_path)) {
      $form_state->setValue('base_path', trim($base_path, '/ '));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('soap_manager.settings')
      // General settings
      ->set('base_path', $form_state->getValue('base_path'))
      ->set('auth_methods', array_filter($form_state->getValue('auth_methods')))
      // Security settings
      ->set('security.request_timeout', $form_state->getValue('request_timeout'))
      ->set('security.max_request_size', $form_state->getValue('max_request_size'))
      ->set('security.rate_limiting', $form_state->getValue('rate_limiting'))
      ->set('security.requests_per_hour', $form_state->getValue('requests_per_hour'))
      // Logging settings
      ->set('logging.log_requests', $form_state->getValue('log_requests'))
      ->set('logging.log_responses', $form_state->getValue('log_responses'))
      ->set('logging.log_errors', $form_state->getValue('log_errors'))
      // Debug settings
      ->set('debug_mode', (bool) $form_state->getValue('debug_mode'))
      ->save();

    parent::submitForm($form, $form_state);

    // Rebuild routes since the base path may have changed.
    \Drupal::service('router.builder')->rebuild();
  }

}
