<?php

namespace Drupal\soap_manager\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\soap_manager\Plugin\SoapResourceManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for adding and editing SOAP endpoint entities.
 */
class SoapManagerForm extends EntityForm {

  /**
   * The SOAP resource plugin manager.
   *
   * @var \Drupal\soap_manager\Plugin\SoapResourceManager
   */
  protected $soapResourceManager;

  /**
   * Constructs a new SoapManagerForm.
   *
   * @param \Drupal\soap_manager\Plugin\SoapResourceManager $soap_resource_manager
   *   The SOAP resource plugin manager.
   */
  public function __construct(SoapResourceManager $soap_resource_manager) {
    $this->soapResourceManager = $soap_resource_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.soap_resource')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager */
    $soap_manager = $this->entity;

    // Set the page title.
    $form['#title'] = $this->operation === 'add'
      ? $this->t('Add SOAP endpoint')
      : $this->t('Edit SOAP endpoint %label', ['%label' => $soap_manager->label()]);

    // Basic endpoint information.
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#maxlength' => 255,
      '#default_value' => $soap_manager->label(),
      '#description' => $this->t('A descriptive name for this SOAP endpoint.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $soap_manager->id(),
      '#machine_name' => [
        'exists' => '\Drupal\soap_manager\Entity\SoapManager::load',
      ],
      '#disabled' => !$soap_manager->isNew(),
    ];

    $form['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path'),
      '#maxlength' => 255,
      '#default_value' => $soap_manager->getPath(),
      '#description' => $this->t('The URL path for this SOAP endpoint (e.g., "soap/my-service"). Do not include leading or trailing slashes.'),
      '#required' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $soap_manager->getDescription(),
      '#description' => $this->t('A brief description of this SOAP endpoint and its purpose.'),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $soap_manager->status() ?? TRUE,
      '#description' => $this->t('Whether this SOAP endpoint is active and can be accessed.'),
    ];

    $form['resources'] = [
      '#type' => 'table',
      '#header' => [
        'enabled' => $this->t('Enabled'),
        'id' => $this->t('ID'),
        'soap_method' => $this->t('SOAP Method'),
        'label' => $this->t('Label'),
        'description' => $this->t('Description'),
      ],
      '#empty' => $this->t('No SOAP resources available. You need to implement SOAP resource plugins to use with this endpoint.'),
      '#default_value' => $soap_manager->getEnabledResources(),
    ];

    foreach ($this->soapResourceManager->getSoapResources() as $definition) {
      /** @var \Drupal\soap_manager\Plugin\SoapResource\SoapResourceInterface $plugin */
      $plugin = $this->soapResourceManager->createInstance($definition['id']);

      $form['resources'][$plugin->getResourceId()]['enabled'] = [
        '#type' => 'checkbox',
        '#default_value' => $soap_manager->hasResource($plugin->getResourceId()),
      ];
      $form['resources'][$plugin->getResourceId()]['id'] = [
        '#type' => 'markup',
        '#markup' => $plugin->getResourceId(),
      ];
      $form['resources'][$plugin->getResourceId()]['soap_method'] = [
        '#type' => 'textfield',
        '#default_value' => $soap_manager->getResourceSoapMethod($plugin->getResourceId()),
      ];
      $form['resources'][$plugin->getResourceId()]['label'] = [
        '#type' => 'markup',
        '#markup' => $plugin->getResourceLabel(),
      ];
      $form['resources'][$plugin->getResourceId()]['description'] = [
        '#type' => 'markup',
        // '#markup' => $plugin->getDescription(),
      ];
    }

    // Authentication settings.
    $form['authentication'] = [
      '#type' => 'details',
      '#title' => $this->t('Authentication'),
      '#description' => $this->t('Configure authentication for this SOAP endpoint.'),
      '#open' => TRUE,
    ];

    $form['authentication']['auth_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require authentication'),
      '#default_value' => $soap_manager->isAuthRequired() ?? FALSE,
      '#description' => $this->t('If checked, users must be authenticated to access this SOAP endpoint.'),
    ];

    // Auth settings will be expanded in future versions to include various auth methods.
    $form['authentication']['auth_settings'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="auth_required"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // WSDL Settings
    $form['wsdl_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('WSDL Settings'),
      '#description' => $this->t('Configure how the WSDL document is provided for this SOAP endpoint.'),
      '#open' => TRUE,
    ];

    $form['wsdl_settings']['wsdl_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom WSDL file path'),
      '#default_value' => $soap_manager->getWsdlPath(),
      '#description' => $this->t('Optional. Specify a path to an existing WSDL file to use instead of auto-generating one. Leave empty to use auto-generated WSDL. This can be an absolute path or a path relative to the Drupal root.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate path - ensure it doesn't have leading/trailing slashes.
    $path = $form_state->getValue('path');
    $path = trim($path, '/');
    $form_state->setValue('path', $path);

    // Remove not enabled resources.
    $resources = $form_state->getValue('resources');

    foreach ($resources as $key => $resource) {
      if (!$resource['enabled']) {
        unset($resources[$key]);
      }
    }

    // Validate at least one enabled resource.
    if (empty($resources)) {
      $form_state->setErrorByName('resources', $this->t('At least one resource must be enabled.'));
    }

    // Check if enabled resources have a SOAP method.
    foreach ($resources as $key => $resource) {
      if ($resource['enabled'] && empty($resource['soap_method'])) {
        $form_state->setError($form['resources'][$key]['soap_method'], $this->t('Enabled resources must have a SOAP method.'));
        break;
      }
    }

    // Override value to store data in format: [resource_id => soap_method].
    foreach ($resources as $resource_id => $resource) {
      $resources[$resource_id] = $resource['soap_method'];
    }

    $form_state->setValue('resources', $resources);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\soap_manager\Entity\SoapManagerInterface $soap_manager */
    $soap_manager = $this->entity;

    $status = $soap_manager->save();

    $t_args = ['%name' => $soap_manager->label()];

    if ($status === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('SOAP endpoint %name has been created.', $t_args));
      $this->logger('soap_manager')->notice('Created new SOAP endpoint %name', $t_args);
    }
    else {
      $this->messenger()->addStatus($this->t('SOAP endpoint %name has been updated.', $t_args));
      $this->logger('soap_manager')->notice('Updated SOAP endpoint %name', $t_args);
    }

    // Redirect to the endpoint list page.
    $form_state->setRedirectUrl(Url::fromRoute('entity.soap_manager.collection'));
  }

}
