<?php

namespace Drupal\soap_manager\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\soap_manager\Annotation\SoapResource;
use Drupal\soap_manager\Plugin\SoapResource\SoapResourceInterface;

/**
 * Provides a plugin manager for SOAP Resource plugins.
 */
class SoapResourceManager extends DefaultPluginManager {

  /**
   * Constructs a new SoapResourceManager.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/SoapResource',
      $namespaces,
      $module_handler,
      SoapResourceInterface::class,
      SoapResource::class
    );

    $this->alterInfo('soap_resource_info');
    $this->setCacheBackend($cache_backend, 'soap_resource_plugins');
  }

  /**
   * Gets a list of all available SOAP resources.
   *
   * @return array
   *   An array of plugin definitions, keyed by plugin ID.
   */
  public function getSoapResources() {
    return $this->getDefinitions();
  }

  /**
   * Invokes a SOAP method on a resource.
   *
   * @param string $resource_id
   *   The resource plugin ID.
   * @param string $method_name
   *   The method name to invoke.
   * @param array $parameters
   *   The parameters to pass to the method.
   *
   * @return mixed
   *   The result of the method call.
   *
   * @throws \Exception
   *   When the resource or method does not exist or another error occurs.
   */
  public function invokeSoapMethod($resource_id, $method_name, array $parameters) {
    if (!$this->hasDefinition($resource_id)) {
      throw new \Exception(sprintf('SOAP resource %s does not exist', $resource_id));
    }

    /** @var \Drupal\soap_manager\Plugin\SoapResource\SoapResourceInterface $instance */
    $instance = $this->createInstance($resource_id);

    if (!$instance->hasMethod($method_name)) {
      throw new \Exception(sprintf('Method %s does not exist on resource %s', $method_name, $resource_id));
    }

    return $instance->invokeMethod($method_name, $parameters);
  }

}
