<?php

namespace Drupal\soap_manager\Routing;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Drupal\Core\Routing\RoutingEvents;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Subscriber for SOAP endpoint routes.
 *
 * Dynamically generates routes for SOAP endpoints based on their configured
 * paths.
 */
class SoapManagerRouteSubscriber extends RouteSubscriberBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new SoapManagerRouteSubscriber.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    // Come after the entity route provider, which has priority 0.
    $events[RoutingEvents::ALTER] = ['onAlterRoutes', -100];
    return $events;
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    // Get all enabled SOAP endpoints.
    try {
      $soap_managers = $this->entityTypeManager
        ->getStorage('soap_manager')
        ->loadByProperties(['status' => TRUE]);
    }
    catch (\Exception $e) {
      // If the entity type doesn't exist yet (during installation), return.
      return;
    }

    // Add a route for each SOAP endpoint.
    foreach ($soap_managers as $soap_manager) {
      // Add the main SOAP endpoint route.
      $path = $soap_manager->getPath();
      if (empty($path)) {
        continue;
      }

      // Create a new route.
      $route = new Route(
        '/' . $path,
        [
          '_controller' => 'Drupal\soap_manager\Controller\SoapManagerController::handleRequest',
          '_title' => $soap_manager->label(),
          'soap_manager' => $soap_manager->id(),
        ],
        [
          '_access' => 'TRUE',
        ]
      );

      $route->setOption('no_cache', TRUE);

      // Add the route to the collection.
      $collection->add('soap_manager.' . $soap_manager->id(), $route);

      // Add a dedicated route for WSDL requests.
      $wsdl_route = new Route(
        '/' . $path . '/wsdl',
        [
          '_controller' => 'Drupal\soap_manager\Controller\SoapManagerController::getWsdlDocument',
          '_title' => $soap_manager->label() . ' WSDL',
          'soap_manager' => $soap_manager->id(),
        ],
        [
          '_access' => 'TRUE',
        ]
      );

      // Add the WSDL route to the collection.
      $collection->add('entity.soap_manager.wsdl', $wsdl_route);
    }
  }

}
