<?php

namespace Drupal\soap_manager\Service;

use Drupal\soap_manager\Plugin\SoapResourceManager;

/**
 * Handles dispatching SOAP method calls to the appropriate resource plugin.
 */
class ServerHandler {

  /**
   * The SOAP resource manager.
   *
   * @var \Drupal\soap_manager\Plugin\SoapResourceManager
   */
  protected $soapResourceManager;

  /**
   * The SOAP security service.
   *
   * @var \Drupal\soap_manager\Service\SoapSecurity
   */
  protected $soapSecurity;

  /**
   * The list of enabled resource plugin IDs.
   *
   * @var array
   */
  protected $enabledResources;

  /**
   * Cache of instantiated resource plugins.
   *
   * @var array
   */
  protected $resourceInstances = [];

  /**
   * Constructs a new ServerHandler object.
   *
   * @param \Drupal\soap_manager\Plugin\SoapResourceManager $soap_resource_manager
   *   The SOAP resource manager.
   * @param \Drupal\soap_manager\Service\SoapSecurity $soap_security
   *   The SOAP security service.
   * @param array $enabled_resources
   *   An array of enabled resource plugin IDs.
   */
  public function __construct(
    SoapResourceManager $soap_resource_manager,
    SoapSecurity $soap_security,
    array $enabled_resources
  ) {
    $this->soapResourceManager = $soap_resource_manager;
    $this->soapSecurity = $soap_security;
    $this->enabledResources = $enabled_resources;
  }

  /**
   * Handles method calls from the SOAP server.
   *
   * This method uses PHP's magic __call() method to catch any SOAP method
   * calls and dispatch them to the appropriate resource plugin.
   *
   * @param string $method_name
   *   The SOAP method name.
   * @param array $arguments
   *   The arguments passed to the method.
   *
   * @return mixed
   *   The method result.
   *
   * @throws \SoapFault
   *   If the method doesn't exist or an error occurs.
   */
  public function __call($method_name, $arguments) {
    // Find which resource plugin can handle this method.
    foreach ($this->enabledResources as $plugin_id => $plugin_soap_method) {
      if ($plugin_soap_method === $method_name) {
        return $this->invokeResourceMethod($plugin_id, $method_name, $arguments);
      }
    }

    // If we get here, the method wasn't found in any enabled resource.
    throw new \SoapFault('Client', "Method '{$method_name}' not found in any enabled resource.");
  }

  /**
   * Invokes a method on a resource plugin.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param string $method_name
   *   The method name.
   * @param array $arguments
   *   The method arguments.
   *
   * @return mixed
   *   The method result.
   *
   * @throws \SoapFault
   *   If an error occurs while invoking the method.
   */
  protected function invokeResourceMethod($plugin_id, $method_name, array $arguments) {
    try {
      // Get or create the resource plugin instance.
      if (!isset($this->resourceInstances[$plugin_id])) {
        $this->resourceInstances[$plugin_id] = $this->soapResourceManager->createInstance($plugin_id);
      }
      $resource = $this->resourceInstances[$plugin_id];

      // Sanitize the input parameters.
      $sanitized_arguments = [];
      foreach ($arguments as $argument) {
        if (is_array($argument)) {
          $sanitized_arguments[] = $this->convertStdClassToArray($argument);
        }
        elseif ($argument instanceof \stdClass) {
          // Convert stdClass to array before sanitizing (recursively)
          $sanitized_arguments[] = $this->convertStdClassToArray($argument);
        }
        else {
          $sanitized_arguments[] = $argument;
        }
      }

      return $resource->processRequest($sanitized_arguments);
    }
    catch (\Exception $e) {
      throw new \SoapFault('Server', $e->getMessage());
    }
  }

  /**
   * Recursively converts stdClass objects to arrays.
   *
   * @param mixed $data
   *   The data to convert.
   *
   * @return mixed
   *   The converted data.
   */
  protected function convertStdClassToArray($data) {
    if (is_object($data) && $data instanceof \stdClass) {
      $data = (array) $data;
    }

    if (is_array($data)) {
      foreach ($data as $key => $value) {
        $data[$key] = $this->convertStdClassToArray($value);
      }
    }

    return $data;
  }

}
