<?php

declare(strict_types=1);

namespace Drupal\soap_manager\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\soap_manager\Entity\SoapManagerLogRequestInterface;
use Drupal\soap_manager\Entity\SoapManagerLogResponseInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for logging SOAP requests and responses.
 */
class SoapLogger {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected FileRepositoryInterface $fileRepository;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The module settings.
   *
   * @var \Drupal\soap_manager\Service\ModuleSettings
   */
  protected ModuleSettings $moduleSettings;

  /**
   * Constructs a new SoapLogger object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\file\FileRepositoryInterface $file_repository
   *   The file repository service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\soap_manager\Service\ModuleSettings $module_settings
   *   The module settings.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
    AccountProxyInterface $current_user,
    RequestStack $request_stack,
    ModuleSettings $module_settings
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem        = $file_system;
    $this->fileRepository    = $file_repository;
    $this->currentUser       = $current_user;
    $this->requestStack      = $request_stack;
    $this->moduleSettings    = $module_settings;
  }

  /**
   * Logs a SOAP request.
   *
   * @param string $endpoint_id
   *   The endpoint ID.
   * @param string $request_xml
   *   The SOAP request XML.
   *
   * @return \Drupal\soap_manager\Entity\SoapManagerLogRequestInterface|null
   *   The created request log entity or NULL if logging is disabled.
   */
  public function logRequest(string $endpoint_id, string $request_xml): ?SoapManagerLogRequestInterface {
    if (!$this->moduleSettings->isRequestLoggingEnabled()) {
      return NULL;
    }

    try {
      // Create the request log entity.
      $storage = $this->entityTypeManager->getStorage('soap_log_request');
      /** @var \Drupal\soap_manager\Entity\SoapManagerLogRequestInterface $log */
      $log = $storage->create([
        'endpoint_id' => $endpoint_id,
        'client_ip'   => $this->getClientIp(),
        'uid'         => $this->currentUser->id(),
      ]);

      // Save the XML as a file.
      $file = $this->saveXmlFile($request_xml, 'soap_logs/requests', 'request');
      if ($file) {
        $log->set('request_xml', ['target_id' => $file->id()]);
      }

      $log->save();

      return $log;
    }
    catch (\Exception $e) {
      \Drupal::logger('soap_manager')->error('Failed to log SOAP request: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Logs a SOAP response.
   *
   * @param int|null $request_log_id
   *   The request log ID.
   * @param string $response_xml
   *   The SOAP response XML.
   * @param int $status_code
   *   The HTTP status code.
   * @param bool $success
   *   Whether the response was successful.
   *
   * @return \Drupal\soap_manager\Entity\SoapManagerLogResponseInterface|null
   *   The created response log entity or NULL if logging is disabled.
   */
  public function logResponse(?int $request_log_id, string $response_xml, int $status_code = 200, bool $success = TRUE): ?SoapManagerLogResponseInterface {
    if (!$this->moduleSettings->isResponseLoggingEnabled()) {
      return NULL;
    }

    try {
      // Create the response log entity.
      $storage = $this->entityTypeManager->getStorage('soap_log_response');
      /** @var \Drupal\soap_manager\Entity\SoapManagerLogResponseInterface $log */
      $log = $storage->create([
        'request_log_id' => $request_log_id,
        'status_code'    => $status_code,
        'success'        => $success,
      ]);

      // Save the XML as a file.
      $file = $this->saveXmlFile($response_xml, 'soap_logs/responses', 'response');
      if ($file) {
        $log->set('response_xml', ['target_id' => $file->id()]);
      }

      $log->save();

      return $log;
    }
    catch (\Exception $e) {
      \Drupal::logger('soap_manager')->error('Failed to log SOAP response: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Saves XML content as a file.
   *
   * @param string $xml_content
   *   The XML content.
   * @param string $directory
   *   The directory to save the file in.
   * @param string $prefix
   *   The filename prefix.
   *
   * @return \Drupal\file\FileInterface|null
   *   The created file entity or NULL on failure.
   */
  protected function saveXmlFile(string $xml_content, string $directory, string $prefix): ?\Drupal\file\FileInterface {
    try {
      // Generate a unique filename.
      $timestamp = time();
      $filename  = sprintf('%s_%s_%s.xml', $prefix, $timestamp, uniqid());

      // Prepare the destination URI.
      $destination = "private://{$directory}/{$filename}";

      // Ensure the directory exists.
      $directory_uri = dirname($destination);
      $this->fileSystem->prepareDirectory($directory_uri, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

      // Save the file.
      $file = $this->fileRepository->writeData($xml_content, $destination, FileSystemInterface::EXISTS_RENAME);

      return $file;
    }
    catch (\Exception $e) {
      \Drupal::logger('soap_manager')->error('Failed to save XML file: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets the client IP address.
   *
   * @return string
   *   The client IP address.
   */
  protected function getClientIp(): string {
    $request = $this->requestStack->getCurrentRequest();
    if ($request) {
      return $request->getClientIp() ?? '127.0.0.1';
    }
    return '127.0.0.1';
  }

}


