<?php

namespace Drupal\soap_manager\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\UserAuthInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Flood\FloodInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Provides security services for SOAP endpoints.
 */
class SoapSecurity {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The user authentication service.
   *
   * @var \Drupal\user\UserAuthInterface
   */
  protected $userAuth;

  /**
   * The module settings service.
   *
   * @var \Drupal\soap_manager\Service\ModuleSettings
   */
  protected $moduleSettings;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The flood service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a SoapSecurity object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\user\UserAuthInterface $user_auth
   *   The user authentication service.
   * @param \Drupal\soap_manager\Service\ModuleSettings $module_settings
   *   The module settings service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    UserAuthInterface $user_auth,
    ModuleSettings $module_settings,
    StateInterface $state,
    FloodInterface $flood,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->userAuth = $user_auth;
    $this->moduleSettings = $module_settings;
    $this->state = $state;
    $this->flood = $flood;
    $this->logger = $logger_factory->get('soap_manager');
  }

  /**
   * Validates the authentication for a SOAP request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param bool $required
   *   Whether authentication is required.
   *
   * @return bool
   *   TRUE if authentication is valid, FALSE otherwise.
   */
  public function validateAuth(Request $request, bool $required = FALSE) {
    // If authentication is not required and the user is already authenticated,
    // return TRUE.
    if (!$required && $this->currentUser->isAuthenticated()) {
      return TRUE;
    }

    // Try to authenticate with WS-Security headers from the SOAP request
    if ($request->getMethod() === 'POST' && $this->validateWsSecurityAuth($request)) {
      return TRUE;
    }

    // If authentication header is present, try to authenticate the user with HTTP Basic Auth.
    if ($request->headers->has('PHP_AUTH_USER')) {
      $username = $request->headers->get('PHP_AUTH_USER');
      $password = $request->headers->get('PHP_AUTH_PW');

      // Authenticate the user.
      $uid = $this->userAuth->authenticate($username, $password);
      if ($uid) {
        // If authentication is successful, set the current user.
        // Note: In a real implementation, you might need to use the user.login
        // service to fully authenticate the user.
        return TRUE;
      }
      else {
        // Log failed authentication attempts.
        $this->logger->warning('Failed SOAP authentication attempt for user @user', [
          '@user' => $username,
        ]);
        return FALSE;
      }
    }

    // If we got here and authentication is required, return FALSE.
    // Otherwise, return TRUE.
    return !$required;
  }

  /**
   * Validates authentication using WS-Security UsernameToken in the SOAP header.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return bool
   *   TRUE if authentication is valid, FALSE otherwise.
   */
  protected function validateWsSecurityAuth(Request $request) {
    $content = $request->getContent();

    // If there's no content, return false
    if (empty($content)) {
      return FALSE;
    }

    // Load XML content
    libxml_use_internal_errors(TRUE);
    $doc = new \DOMDocument();
    $result = $doc->loadXML($content);

    if (!$result) {
      libxml_clear_errors();
      return FALSE;
    }

    // Extract username and password from the WS-Security header
    $credentials = $this->extractWsSecurityCredentials($doc);

    if (!$credentials) {
      return FALSE;
    }

    // Authenticate the user using extracted credentials
    $username = $credentials['username'];
    $password = $credentials['password'];

    $uid = $this->userAuth->authenticate($username, $password);

    if ($uid) {
      return TRUE;
    }

    // Log failed authentication attempts
    $this->logger->warning('Failed WS-Security authentication attempt for user @user', [
      '@user' => $username,
    ]);

    return FALSE;
  }

  /**
   * Extracts WS-Security UsernameToken credentials from a SOAP message.
   *
   * @param \DOMDocument $doc
   *   The SOAP XML document.
   *
   * @return array|false
   *   An array containing 'username' and 'password', or FALSE if not found.
   */
  protected function extractWsSecurityCredentials(\DOMDocument $doc) {
    // Register the namespaces
    $namespaces = [
      'SOAP-ENV' => 'http://schemas.xmlsoap.org/soap/envelope/',
      'wsse' => 'http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd',
    ];

    // Create a DOMXPath object
    $xpath = new \DOMXPath($doc);

    // Register the namespaces with xpath
    foreach ($namespaces as $prefix => $ns) {
      $xpath->registerNamespace($prefix, $ns);
    }

    // Query for the Security header
    $security = $xpath->query('//SOAP-ENV:Header/wsse:Security');

    if ($security->length === 0) {
      return FALSE;
    }

    // Query for UsernameToken within Security header
    $usernameToken = $xpath->query('./wsse:UsernameToken', $security->item(0));

    if ($usernameToken->length === 0) {
      return FALSE;
    }

    // Extract username
    $username = $xpath->query('./wsse:Username', $usernameToken->item(0));

    if ($username->length === 0) {
      return FALSE;
    }

    // Extract password
    $password = $xpath->query('./wsse:Password', $usernameToken->item(0));

    if ($password->length === 0) {
      return FALSE;
    }

    return [
      'username' => $username->item(0)->nodeValue,
      'password' => $password->item(0)->nodeValue,
    ];
  }

}
