<?php

namespace Drupal\soap_manager\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\soap_manager\Plugin\SoapResourceManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\soap_manager\Entity\SoapManagerInterface;
use Drupal\soap_manager\Service\SoapSecurity;
use Drupal\soap_manager\Service\ModuleSettings;
use Drupal\soap_manager\Service\SoapLogger;

/**
 * Service for handling SOAP requests.
 */
class SoapServer {

  /**
   * The SOAP resource plugin manager.
   *
   * @var \Drupal\soap_manager\Plugin\SoapResourceManager
   */
  protected $soapResourceManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The SOAP security service.
   *
   * @var \Drupal\soap_manager\Service\SoapSecurity
   */
  protected $soapSecurity;

  /**
   * The module settings.
   *
   * @var \Drupal\soap_manager\Service\ModuleSettings
   */
  protected $moduleSettings;

  /**
   * The SOAP logger service.
   *
   * @var \Drupal\soap_manager\Service\SoapLogger
   */
  protected $soapLogger;

  /**
   * Constructs a new SoapServer.
   *
   * @param \Drupal\soap_manager\Plugin\SoapResourceManager $soap_resource_manager
   *   The SOAP resource plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\soap_manager\Service\SoapSecurity $soap_security
   *   The SOAP security service.
   * @param \Drupal\soap_manager\Service\ModuleSettings $module_settings
   *   The module settings.
   * @param \Drupal\soap_manager\Service\SoapLogger $soap_logger
   *   The SOAP logger service.
   */
  public function __construct(
    SoapResourceManager $soap_resource_manager,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    SoapSecurity $soap_security,
    ModuleSettings $module_settings,
    SoapLogger $soap_logger
  ) {
    $this->soapResourceManager = $soap_resource_manager;
    $this->logger = $logger_factory->get('soap_manager');
    $this->entityTypeManager = $entity_type_manager;
    $this->soapSecurity = $soap_security;
    $this->moduleSettings = $module_settings;
    $this->soapLogger = $soap_logger;
  }

  /**
   * Handles the SOAP request using the appropriate plugin.
   *
   * @param string $endpoint_id
   *   The endpoint ID.
   * @param string $wsdl_url
   *   The URL to the WSDL document.
   * @param array $options
   *   Additional options for the SOAP server.
   * @param string $soap_request
   *   The raw SOAP request data.
   *
   * @return string
   *   The SOAP response.
   */
  public function handleRequest($endpoint_id, $wsdl_url, array $options, $soap_request = NULL) {
    // Get the endpoint entity.
    /** @var \Drupal\soap_manager\Entity\SoapManagerInterface $endpoint */
    $endpoint = $this->entityTypeManager->getStorage('soap_manager')->load($endpoint_id);
    if (!$endpoint) {
      $this->logger->error('Endpoint not found: @id', ['@id' => $endpoint_id]);
      return $this->createSoapFault('Server', 'Endpoint Not Found');
    }

    // Check if the endpoint is enabled.
    if (!$endpoint->status()) {
      $this->logger->error('Endpoint disabled: @id', ['@id' => $endpoint_id]);
      return $this->createSoapFault('Server', 'Endpoint Disabled');
    }

    // If there's no request data, attempt to get it from the PHP input stream.
    if (empty($soap_request)) {
      $soap_request = file_get_contents('php://input');
      if (empty($soap_request)) {
        $this->logger->error('Empty SOAP request received');
        return $this->createSoapFault('Client', 'Empty SOAP request');
      }
    }

    try {
      // Extract SOAP method from request.
      // Log the incoming request if enabled
      $request_log = NULL;
      if ($this->moduleSettings->isRequestLoggingEnabled()) {
        $request_log = $this->soapLogger->logRequest($endpoint_id, $soap_request);
        $this->logger->notice('SOAP request to @endpoint: @request', [
          '@endpoint' => $endpoint->label(),
          '@request' => $soap_request,
        ]);
      }

      // Get enabled resources for this endpoint.
      $enabled_resources = $endpoint->getEnabledResources();
      if (empty($enabled_resources)) {
        $this->logger->error('No enabled resources for endpoint: @id', ['@id' => $endpoint_id]);
        $fault_response = $this->createSoapFault('Server', 'No SOAP resources are enabled for this endpoint.');

        // Log fault response.
        if ($this->moduleSettings->isResponseLoggingEnabled() && $request_log) {
          $this->soapLogger->logResponse($request_log->id(), $fault_response, 500, FALSE);
        }

        return $fault_response;
      }

      // Disable the WSDL cache so it's not stored in memory or on disk.
      ini_set("soap.wsdl_cache_enabled", 0);

      // Create a ServerHandler that will dispatch to our plugin resources.
      $server_handler = new ServerHandler($this->soapResourceManager, $this->soapSecurity, $enabled_resources);

      // Create the SOAP server.
      $server = new \SoapServer($wsdl_url, $options);
      $server->setObject($server_handler);

      // Turn output buffering on.
      ob_start();
      // Handle the SOAP request.
      $server->handle($soap_request);
      // Get the buffers contents.
      $soap_response = ob_get_contents();
      // Removes topmost output buffer.
      ob_end_clean();

      // Remove any Content-Length header to prevent duplicate headers
      // since Symfony's Response object will add this header automatically
      if (function_exists('header_remove')) {
        header_remove('Content-Length');
      }

      // Log the response if enabled
      if ($this->moduleSettings->isResponseLoggingEnabled() && $request_log) {
        $is_success = !str_contains($soap_response, 'SOAP-ENV:Fault');
        $this->soapLogger->logResponse($request_log->id(), $soap_response, 200, $is_success);
        $this->logger->notice('SOAP response from @endpoint: @response', [
          '@endpoint' => $endpoint->label(),
          '@response' => $soap_response,
        ]);
      }

      return $soap_response;
    }
    catch (\Exception $e) {
      // Log the error with detailed information
      $this->logger->error('Error processing SOAP request: @error. Trace: @trace', [
        '@error' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);

      // Create a fault response.
      $fault_response = $this->createSoapFault('Server', 'Server Error: ' . $e->getMessage());

      // Log fault response if we have a request log.
      if ($this->moduleSettings->isResponseLoggingEnabled() && isset($request_log) && $request_log) {
        $this->soapLogger->logResponse($request_log->id(), $fault_response, 500, FALSE);
      }

      return $fault_response;
    }
  }

  /**
   * Generates a SOAP fault response.
   *
   * @param string $code
   *   The fault code.
   * @param string $message
   *   The fault message.
   *
   * @return string
   *   The SOAP fault response.
   */
  protected function createSoapFault($code, $message) {
    $response = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $response .= '<SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/">' . "\n";
    $response .= '  <SOAP-ENV:Body>' . "\n";
    $response .= '    <SOAP-ENV:Fault>' . "\n";
    $response .= '      <faultcode>' . htmlspecialchars($code) . '</faultcode>' . "\n";
    $response .= '      <faultstring>' . htmlspecialchars($message) . '</faultstring>' . "\n";
    $response .= '    </SOAP-ENV:Fault>' . "\n";
    $response .= '  </SOAP-ENV:Body>' . "\n";
    $response .= '</SOAP-ENV:Envelope>';
    return $response;
  }

}
