<?php

namespace Drupal\soap_manager;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Url;

/**
 * Provides a listing of SOAP Endpoint entities.
 */
class SoapManagerListBuilder extends ConfigEntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('SOAP Endpoint');
    $header['id'] = $this->t('Machine name');
    $header['path'] = $this->t('Path');
    $header['status'] = $this->t('Status');
    $header['auth'] = $this->t('Authentication');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\soap_manager\Entity\SoapManagerInterface $entity */
    $row['label'] = $entity->label();
    $row['id'] = $entity->id();
    $row['path'] = '/' . $entity->getPath();

    // Status indicator.
    $status = $entity->status() ? $this->t('Enabled') : $this->t('Disabled');
    $row['status'] = [
      'data' => [
        '#markup' => $status,
      ],
    ];

    // Auth indicator.
    $auth = $entity->isAuthRequired() ? $this->t('Required') : $this->t('None');
    $row['auth'] = [
      'data' => [
        '#markup' => $auth,
      ],
    ];

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    // Add a link to view the SOAP endpoint information page.
    $operations['view'] = [
      'title' => $this->t('View'),
      'weight' => 10,
      'url' => Url::fromRoute('entity.soap_manager.canonical', [
        'soap_manager' => $entity->id(),
      ]),
    ];

    // Add a link to view the WSDL document.
    $operations['wsdl'] = [
      'title' => $this->t('WSDL'),
      'weight' => 15,
      'url' => Url::fromRoute('entity.soap_manager.wsdl', [
        'soap_manager' => $entity->id(),
      ]),
    ];

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();

    // Add descriptive text to the top of the list.
    $build['description'] = [
      '#markup' => $this->t('
        <p>SOAP endpoints expose web services using the SOAP protocol.</p>
        <p>Each endpoint can include multiple SOAP resources and methods, which are defined by plugins.</p>
      '),
      '#weight' => -10,
    ];

    return $build;
  }

}
