<?php

declare(strict_types=1);

namespace Drupal\soap_manager;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Defines a class to build a listing of SOAP request log entities.
 */
class SoapManagerLogRequestListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id']           = $this->t('ID');
    $header['endpoint_id']  = $this->t('Endpoint');
    $header['client_ip']    = $this->t('Client IP');
    $header['response_log_id'] = $this->t('Response Log ID');
    $header['user']         = $this->t('User');
    $header['created']      = $this->t('Created');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\soap_manager\Entity\SoapManagerLogRequestInterface $entity */
    $row['id'] = $entity->id();
    $row['endpoint_id'] = $entity->getEndpointId();
    $row['client_ip'] = $entity->getClientIp();

    // Link to request log if it exists.
    $response_log_id = $entity->getResponseLogId();
    if ($response_log_id) {
      $url = Url::fromRoute('entity.soap_log_response.canonical', [
        'soap_log_response' => $response_log_id,
      ]);
      $row['response_log_id'] = Link::fromTextAndUrl($response_log_id, $url)->toString();
    }
    else {
      $row['response_log_id'] = $this->t('N/A');
    }

    // Display user.
    $owner = $entity->getOwner();
    if ($owner) {
      $row['user'] = $owner->getDisplayName();
    }
    else {
      $row['user'] = $this->t('Anonymous');
    }

    $row['created'] = \Drupal::service('date.formatter')->format($entity->getCreatedTime(), 'short');

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);

    // Add a view operation.
    $operations['view'] = [
      'title' => $this->t('View'),
      'weight' => 0,
      'url' => $entity->toUrl('canonical'),
    ];

    return $operations;
  }

  /**
   * Returns a query object for loading entity IDs from the storage.
   *
   * @return \Drupal\Core\Entity\Query\QueryInterface
   *   A query object used to load entity IDs.
   */
  protected function getEntityListQuery(): QueryInterface {
    $query = $this->getStorage()->getQuery()
      ->accessCheck(TRUE)
      ->sort($this->entityType->getKey(static::SORT_KEY), 'DESC');

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }
    return $query;
  }

}


