<?php

declare(strict_types=1);

namespace Drupal\soap_manager;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Defines a class to build a listing of SOAP response log entities.
 */
class SoapManagerLogResponseListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id']              = $this->t('ID');
    $header['request_log_id']  = $this->t('Request Log ID');
    $header['status_code']     = $this->t('Status Code');
    $header['success']         = $this->t('Success');
    $header['created']         = $this->t('Created');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\soap_manager\Entity\SoapManagerLogResponseInterface $entity */
    $row['id'] = $entity->id();

    // Link to request log if it exists.
    $request_log_id = $entity->getRequestLogId();
    if ($request_log_id) {
      $url = Url::fromRoute('entity.soap_log_request.canonical', [
        'soap_log_request' => $request_log_id,
      ]);
      $row['request_log_id'] = Link::fromTextAndUrl($request_log_id, $url)->toString();
    }
    else {
      $row['request_log_id'] = $this->t('N/A');
    }

    $row['status_code'] = $entity->getStatusCode() ?? $this->t('N/A');
    $row['success'] = $entity->isSuccess() ? $this->t('Yes') : $this->t('No');
    $row['created'] = \Drupal::service('date.formatter')->format($entity->getCreatedTime(), 'short');

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);

    // Add a view operation.
    $operations['view'] = [
      'title' => $this->t('View'),
      'weight' => 0,
      'url' => $entity->toUrl('canonical'),
    ];

    return $operations;
  }

  /**
   * Returns a query object for loading entity IDs from the storage.
   *
   * @return \Drupal\Core\Entity\Query\QueryInterface
   *   A query object used to load entity IDs.
   */
  protected function getEntityListQuery(): QueryInterface {
    $query = $this->getStorage()->getQuery()
      ->accessCheck(TRUE)
      ->sort($this->entityType->getKey(static::SORT_KEY), 'DESC');

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }
    return $query;
  }

}


