<?php

namespace Drupal\soap_manager;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\soap_manager\Entity\SoapManagerInterface;
use Drupal\soap_manager\Plugin\SoapResourceManager;

/**
 * Generates WSDL documents for SOAP endpoints.
 */
class WsdlGenerator {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The SOAP resource plugin manager.
   *
   * @var \Drupal\soap_manager\Plugin\SoapResourceManager
   */
  protected $soapResourceManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new WsdlGenerator.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\soap_manager\Plugin\SoapResourceManager $soap_resource_manager
   *   The SOAP resource plugin manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    SoapResourceManager $soap_resource_manager,
    AccountProxyInterface $current_user,
    RendererInterface $renderer
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->soapResourceManager = $soap_resource_manager;
    $this->currentUser = $current_user;
    $this->renderer = $renderer;
  }

  /**
   * Generates a WSDL document for a SOAP endpoint.
   *
   * @param \Drupal\soap_manager\Entity\SoapManagerInterface $endpoint
   *   The SOAP endpoint.
   * @param string $base_url
   *   The base URL of the site.
   *
   * @return string
   *   The WSDL document.
   */
  public function generateWsdl(SoapManagerInterface $endpoint, $base_url) {
    // Define XML namespaces.
    $namespace = $base_url . '/' . $endpoint->getPath();
    $service_name = $endpoint->label();
    $targetNamespace = $namespace;

    // Create the XML document.
    $wsdl = new \DOMDocument('1.0', 'UTF-8');
    $wsdl->formatOutput = TRUE;

    // Create the root element.
    $definitions = $wsdl->createElementNS('http://schemas.xmlsoap.org/wsdl/', 'definitions');
    $wsdl->appendChild($definitions);

    // Set attributes for the root element.
    $definitions->setAttribute('name', $service_name);
    $definitions->setAttribute('targetNamespace', $targetNamespace);
    $definitions->setAttribute('xmlns:tns', $targetNamespace);
    $definitions->setAttribute('xmlns:soap', 'http://schemas.xmlsoap.org/wsdl/soap/');
    $definitions->setAttribute('xmlns:xsd', 'http://www.w3.org/2001/XMLSchema');
    $definitions->setAttribute('xmlns:soapenc', 'http://schemas.xmlsoap.org/soap/encoding/');
    $definitions->setAttribute('xmlns:wsdl', 'http://schemas.xmlsoap.org/wsdl/');
    // Add WS-Security Policy namespaces
    $definitions->setAttribute('xmlns:wsp', 'http://www.w3.org/ns/ws-policy');
    $definitions->setAttribute('xmlns:sp', 'http://docs.oasis-open.org/ws-sx/ws-securitypolicy/200702');

    // Create the types element.
    $types = $wsdl->createElement('types');
    $definitions->appendChild($types);

    $schema = $wsdl->createElementNS('http://www.w3.org/2001/XMLSchema', 'xsd:schema');
    $schema->setAttribute('targetNamespace', $targetNamespace);
    $types->appendChild($schema);

    // Get all enabled resources for this endpoint.
    $enabled_resources = $endpoint->getEnabledResources();
    $methods = [];

    // Collect all SOAP methods from enabled resources.
    foreach ($enabled_resources as $resource_id) {
      if (!$this->soapResourceManager->hasDefinition($resource_id)) {
        continue;
      }

      $resource_instance = $this->soapResourceManager->createInstance($resource_id);
      $resource_methods = $resource_instance->getSoapMethodsInfo();

      foreach ($resource_methods as $method_name => $method_info) {
        $methods[] = SoapMethod::createFromInfo($resource_id, $method_name, $method_info);
      }
    }

    // Generate message, portType, binding, and service elements.
    $this->addMessages($wsdl, $definitions, $methods);
    $this->addPortType($wsdl, $definitions, $methods, $service_name);
    $this->addBinding($wsdl, $definitions, $methods, $service_name);
    $this->addService($wsdl, $definitions, $base_url, $endpoint->getPath(), $service_name);

    // Add WS-Security policy if authentication is required
    if ($endpoint->isAuthRequired()) {
      $this->addWsSecurityPolicy($wsdl, $definitions);
    }

    return $wsdl->saveXML();
  }

  /**
   * Adds message elements to the WSDL document.
   *
   * @param \DOMDocument $wsdl
   *   The WSDL document.
   * @param \DOMElement $definitions
   *   The definitions element.
   * @param \Drupal\soap_manager\SoapMethod[] $methods
   *   The SOAP methods.
   */
  protected function addMessages(\DOMDocument $wsdl, \DOMElement $definitions, array $methods) {
    foreach ($methods as $method) {
      // Add request message.
      $request_message = $wsdl->createElement('message');
      $request_message->setAttribute('name', $method->getName() . 'Request');
      $definitions->appendChild($request_message);

      // Add parts for parameters.
      foreach ($method->getParameters() as $param_name => $param_info) {
        $part = $wsdl->createElement('part');
        $part->setAttribute('name', $param_name);
        $part->setAttribute('type', $this->mapType($param_info['type']));
        $request_message->appendChild($part);
      }

      // Add response message.
      $response_message = $wsdl->createElement('message');
      $response_message->setAttribute('name', $method->getName() . 'Response');
      $definitions->appendChild($response_message);

      // Add part for return value.
      $return_part = $wsdl->createElement('part');
      $return_part->setAttribute('name', 'return');
      $return_part->setAttribute('type', $this->mapType($method->getReturnType()));
      $response_message->appendChild($return_part);
    }
  }

  /**
   * Adds a portType element to the WSDL document.
   *
   * @param \DOMDocument $wsdl
   *   The WSDL document.
   * @param \DOMElement $definitions
   *   The definitions element.
   * @param \Drupal\soap_manager\SoapMethod[] $methods
   *   The SOAP methods.
   * @param string $service_name
   *   The name of the service.
   */
  protected function addPortType(\DOMDocument $wsdl, \DOMElement $definitions, array $methods, $service_name) {
    $port_type = $wsdl->createElement('portType');
    $port_type->setAttribute('name', $service_name . 'PortType');
    $definitions->appendChild($port_type);

    foreach ($methods as $method) {
      $operation = $wsdl->createElement('operation');
      $operation->setAttribute('name', $method->getName());
      $port_type->appendChild($operation);

      // Add documentation.
      if ($method->getDescription()) {
        $documentation = $wsdl->createElement('documentation');
        $documentation->appendChild($wsdl->createTextNode($method->getDescription()));
        $operation->appendChild($documentation);
      }

      // Add input.
      $input = $wsdl->createElement('input');
      $input->setAttribute('message', 'tns:' . $method->getName() . 'Request');
      $operation->appendChild($input);

      // Add output.
      $output = $wsdl->createElement('output');
      $output->setAttribute('message', 'tns:' . $method->getName() . 'Response');
      $operation->appendChild($output);
    }
  }

  /**
   * Adds a binding element to the WSDL document.
   *
   * @param \DOMDocument $wsdl
   *   The WSDL document.
   * @param \DOMElement $definitions
   *   The definitions element.
   * @param \Drupal\soap_manager\SoapMethod[] $methods
   *   The SOAP methods.
   * @param string $service_name
   *   The name of the service.
   */
  protected function addBinding(\DOMDocument $wsdl, \DOMElement $definitions, array $methods, $service_name) {
    $binding = $wsdl->createElement('binding');
    $binding->setAttribute('name', $service_name . 'Binding');
    $binding->setAttribute('type', 'tns:' . $service_name . 'PortType');
    $definitions->appendChild($binding);

    $soap_binding = $wsdl->createElementNS('http://schemas.xmlsoap.org/wsdl/soap/', 'soap:binding');
    $soap_binding->setAttribute('style', 'rpc');
    $soap_binding->setAttribute('transport', 'http://schemas.xmlsoap.org/soap/http');
    $binding->appendChild($soap_binding);

    // Add reference to the WS-Security policy
    $policyReference = $wsdl->createElementNS('http://www.w3.org/ns/ws-policy', 'wsp:PolicyReference');
    $policyReference->setAttribute('URI', '#UsernameTokenPolicy');
    $binding->appendChild($policyReference);

    foreach ($methods as $method) {
      $operation = $wsdl->createElement('operation');
      $operation->setAttribute('name', $method->getName());
      $binding->appendChild($operation);

      $soap_operation = $wsdl->createElementNS('http://schemas.xmlsoap.org/wsdl/soap/', 'soap:operation');
      $soap_operation->setAttribute('soapAction', $method->getName());
      $operation->appendChild($soap_operation);

      // Add input.
      $input = $wsdl->createElement('input');
      $operation->appendChild($input);

      $soap_body = $wsdl->createElementNS('http://schemas.xmlsoap.org/wsdl/soap/', 'soap:body');
      $soap_body->setAttribute('use', 'encoded');
      $soap_body->setAttribute('namespace', 'urn:' . $service_name);
      $soap_body->setAttribute('encodingStyle', 'http://schemas.xmlsoap.org/soap/encoding/');
      $input->appendChild($soap_body);

      // Add output.
      $output = $wsdl->createElement('output');
      $operation->appendChild($output);

      $soap_body = $wsdl->createElementNS('http://schemas.xmlsoap.org/wsdl/soap/', 'soap:body');
      $soap_body->setAttribute('use', 'encoded');
      $soap_body->setAttribute('namespace', 'urn:' . $service_name);
      $soap_body->setAttribute('encodingStyle', 'http://schemas.xmlsoap.org/soap/encoding/');
      $output->appendChild($soap_body);
    }
  }

  /**
   * Adds a service element to the WSDL document.
   *
   * @param \DOMDocument $wsdl
   *   The WSDL document.
   * @param \DOMElement $definitions
   *   The definitions element.
   * @param string $base_url
   *   The base URL of the site.
   * @param string $path
   *   The path of the SOAP endpoint.
   * @param string $service_name
   *   The name of the service.
   */
  protected function addService(\DOMDocument $wsdl, \DOMElement $definitions, $base_url, $path, $service_name) {
    $service = $wsdl->createElement('service');
    $service->setAttribute('name', $service_name . 'Service');
    $definitions->appendChild($service);

    $port = $wsdl->createElement('port');
    $port->setAttribute('name', $service_name . 'Port');
    $port->setAttribute('binding', 'tns:' . $service_name . 'Binding');
    $service->appendChild($port);

    $soap_address = $wsdl->createElementNS('http://schemas.xmlsoap.org/wsdl/soap/', 'soap:address');
    $soap_address->setAttribute('location', $base_url . '/' . $path);
    $port->appendChild($soap_address);
  }

  /**
   * Adds WS-Security policy to the WSDL.
   *
   * @param \DOMDocument $wsdl
   *   The WSDL document.
   * @param \DOMElement $definitions
   *   The definitions element.
   */
  protected function addWsSecurityPolicy(\DOMDocument $wsdl, \DOMElement $definitions) {
    // Create the policy element
    $policy = $wsdl->createElementNS('http://www.w3.org/ns/ws-policy', 'wsp:Policy');
    $policy->setAttribute('wsu:Id', 'UsernameTokenPolicy');
    $policy->setAttribute('xmlns:wsu', 'http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd');
    $definitions->appendChild($policy);

    // Create the supporting tokens element
    $supportingTokens = $wsdl->createElementNS('http://docs.oasis-open.org/ws-sx/ws-securitypolicy/200702', 'sp:SupportingTokens');
    $policy->appendChild($supportingTokens);

    // Create nested policy
    $nestedPolicy = $wsdl->createElementNS('http://www.w3.org/ns/ws-policy', 'wsp:Policy');
    $supportingTokens->appendChild($nestedPolicy);

    // Create username token element
    $usernameToken = $wsdl->createElementNS('http://docs.oasis-open.org/ws-sx/ws-securitypolicy/200702', 'sp:UsernameToken');
    $usernameToken->setAttribute('sp:IncludeToken', 'http://docs.oasis-open.org/ws-sx/ws-securitypolicy/200702/IncludeToken/AlwaysToRecipient');
    $nestedPolicy->appendChild($usernameToken);

    // Create inner policy
    $innerPolicy = $wsdl->createElementNS('http://www.w3.org/ns/ws-policy', 'wsp:Policy');
    $usernameToken->appendChild($innerPolicy);

    // Add WssUsernameToken11
    $wssUsernameToken = $wsdl->createElementNS('http://docs.oasis-open.org/ws-sx/ws-securitypolicy/200702', 'sp:WssUsernameToken11');
    $innerPolicy->appendChild($wssUsernameToken);
  }

  /**
   * Maps a PHP type to an XML Schema type.
   *
   * @param string $php_type
   *   The PHP type.
   *
   * @return string
   *   The XML Schema type.
   */
  protected function mapType($php_type) {
    $map = [
      'string' => 'xsd:string',
      'int' => 'xsd:int',
      'integer' => 'xsd:integer',
      'float' => 'xsd:float',
      'double' => 'xsd:double',
      'boolean' => 'xsd:boolean',
      'bool' => 'xsd:boolean',
      'array' => 'xsd:anyType',
      'object' => 'xsd:anyType',
      'mixed' => 'xsd:anyType',
      'void' => 'xsd:void',
      'date' => 'xsd:date',
      'datetime' => 'xsd:dateTime',
    ];

    return isset($map[$php_type]) ? $map[$php_type] : 'xsd:string';
  }

}
