<?php

namespace Drupal\social_group;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Defines the helper service interface.
 *
 * @package Drupal\social_group
 */
interface SocialGroupHelperServiceInterface {

  /**
   * SocialGroupHelperService constructor.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation
   *   The string translation service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    Connection $connection,
    ModuleHandlerInterface $module_handler,
    TranslationInterface $translation,
    EntityTypeManagerInterface $entity_type_manager,
    RendererInterface $renderer,
  );

  /**
   * Gets a description as HTML markup for the selected visibility option.
   *
   * @param string $key
   *   The visibility option name.
   * @param string $hook
   *   The name of hook for altering description.
   */
  public function description(string $key, string $hook): string;

  /**
   * Returns a group id from a entity (post, node).
   *
   * The group that this entity belongs to or NULL if the entity doesn't belong
   * to any group.
   *
   * @param array $entity
   *   The entity in the form of an entity reference array to get the group for.
   * @param bool $read_cache
   *   (optional) Whether the per-request cache should be used. This should only
   *   be disabled if you know that the group for the entity has changed because
   *   disabling this can have serious performance implications. Setting this to
   *   FALSE will update the cache for subsequent calls. Defaults to TRUE.
   */
  public function getGroupFromEntity(array $entity, bool $read_cache = TRUE): ?int;

  /**
   * Returns the statically cached group members form the current group.
   *
   * @return array
   *   All group members as array with value user->id().
   */
  public static function getCurrentGroupMembers();

  /**
   * Get all group memberships for a certain user.
   *
   * @param int $uid
   *   The UID for which we fetch the groups it is member of.
   *
   * @return array
   *   List of group IDs the user is member of.
   */
  public function getAllGroupsForUser(int $uid);

  /**
   * Count all group memberships for a certain user.
   *
   * @param string $uid
   *   The UID for which we fetch the groups it is member of.
   *
   * @return int
   *   Count of groups a user is a member of.
   */
  public function countGroupMembershipsForUser(string $uid): int;

  /**
   * Get the add group URL for given user.
   *
   * This returns either /group/add or /group/add/{group_type}
   * depending upon the permission of the user to create group.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Url|null
   *   URL of the group add page.
   *
   * @deprecated Use the default route with 'flexible_group' as param directly.
   *
   * @see https://www.drupal.org/node/3420070
   */
  public function getGroupsToAddUrl(AccountInterface $account);

  /**
   * Provides a field for potential members.
   */
  public function addMemberFormField(): array;

}
