<?php

namespace Drupal\social_auth_decoupled\User;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\social_auth\User\SocialAuthUser;
use Drupal\social_auth\User\UserAuthenticator;
use Drupal\user\UserInterface;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Manages Drupal authentication tasks for Social Auth.
 */
class UserAuthenticatorHttp extends UserAuthenticator {

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * Injects the csrf_token service.
   *
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The csrf_token.
   */
  public function setCsrfToken(CsrfTokenGenerator $csrf_token) {
    $this->csrfToken = $csrf_token;
  }

  /**
   * Creates and/or authenticates an user.
   *
   * @param string $name
   *   The user's name.
   * @param string $email
   *   The user's email address.
   * @param string $provider_user_id
   *   The unique id returned by the user.
   * @param string $token
   *   The access token for making additional API calls.
   * @param string|bool $picture_url
   *   The user's picture.
   * @param string $data
   *   The additional user_data to be stored in database.
   *
   * @return array|\Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Response Array.
   */
  public function authenticateUser($name, $email, $provider_user_id, $token, $picture_url = FALSE, $data = '') {

    // Checks for record in Social Auth entity.
    $user_id = $this->userManager->getDrupalUserId($provider_user_id);

    // If user is already authenticated.
    if ($this->currentUser->isAuthenticated()) {

      // If no record for provider exists.
      if ($user_id === FALSE) {
        return $this->associateNewProvider($provider_user_id, $token, $data);
      }
      // User is authenticated and provider is already associated.
      else {
        return ['message' => 'You are already an authenticated user, no need login again, but the new social user info has been associate with you just now.'];
      }
    }

    // If user previously authorized the provider, load user through provider.
    if ($user_id) {
      $response_data = $this->authenticateWithProvider($user_id);
      if (!empty($response_data)) {
        return $response_data;
      }
    }

    // Try to authenticate user using email address.
    if ($email) {
      // If authentication with email was successful.
      $response_data = $this->authenticateWithEmail($email, $provider_user_id, $token, $data);
      if (!empty($response_data)) {
        return $response_data;
      }
    }

    $user = new SocialAuthUser($name, $email, $provider_user_id, $token, $picture_url, $data);

    // At this point, create a new user.
    $drupal_user = $this->userManager->createNewUser($user);

    return $this->authenticateNewUser($drupal_user);
  }

  /**
   * Associates an existing user with a new provider.
   *
   * @param string $provider_user_id
   *   The unique id returned by the user.
   * @param string $token
   *   The access token for making additional API calls.
   * @param string $data
   *   The additional user_data to be stored in database.
   */
  public function associateNewProvider($provider_user_id, $token, $data) {
    if ($this->userManager->addUserRecord($this->currentUser->id(), $provider_user_id, $token, $data)) {
      return ['message' => 'You are already an authenticated user, no need login again, but the new social user info has been associate with you just now.'];
    }

    throw new BadRequestHttpException($this->t('New provider could not be associated.'));
  }

  /**
   * Authenticates user using provider.
   *
   * @param int $user_id
   *   The Drupal user id.
   *
   * @return array|\Symfony\Component\HttpKernel\Exception\BadRequestHttpException|false
   *   True is user provider could be associated.
   *   False otherwise.
   */
  public function authenticateWithProvider($user_id) {
    try {
      // Load the user by their Drupal user id.
      $drupal_user = $this->userManager->loadUserByProperty('uid', $user_id);

      if ($drupal_user) {
        // Authenticates and redirect existing user.
        return $this->authenticateExistingUser($drupal_user);
      }

      return FALSE;
    }
    catch (\Exception $ex) {
      throw new BadRequestHttpException($this->t('Failed to authenticate user. Exception: @message', ['@message' => $ex->getMessage()]));
    }
  }

  /**
   * Authenticates and redirects existing users in authentication process.
   *
   * @param \Drupal\user\UserInterface $drupal_user
   *   User object to authenticate.
   *
   * @return array|\Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Response info.
   */
  public function authenticateExistingUser(UserInterface $drupal_user) {
    // If Admin (user 1) can not authenticate.
    if ($this->isAdminDisabled($drupal_user)) {
      $this->nullifySessionKeys();
      throw new BadRequestHttpException($this->t('Authentication for Admin (user 1) is disabled.'));
    }

    // If user can not login because of their role.
    $disabled_role = $this->isUserRoleDisabled($drupal_user);
    if ($disabled_role) {
      throw new BadRequestHttpException($this->t("Authentication for '@role' role is disabled.", ['@role' => $disabled_role]));
    }

    // If user could be logged in.
    $response_data = $this->loginUser($drupal_user);
    if ($response_data) {
      return $response_data;
    }

    else {
      throw new BadRequestHttpException($this->t('Your account has not been approved yet or might have been canceled, please contact the administrator.'));
    }
  }

  /**
   * Authenticates and redirects new users in authentication process.
   *
   * @param \Drupal\user\UserInterface|null $drupal_user
   *   User object to login.
   */
  public function authenticateNewUser(UserInterface $drupal_user = NULL) {

    // If it's a valid Drupal user.
    if ($drupal_user) {

      // If the account needs admin approval.
      if ($this->isApprovalRequired()) {
        throw new BadRequestHttpException($this->t("Your account was created, but it needs administrator's approval."));
      }

      return $this->loginUser($drupal_user);
    }

    if (!$this->isRegistrationDisabled()) {
      throw new BadRequestHttpException($this->t('You could not be authenticated. Contact site administrator.'));
    }

    $this->nullifySessionKeys();
    throw new BadRequestHttpException($this->t('Sth wrong. Contact site administrator.'));
  }

  /**
   * Authenticates user by email address.
   *
   * @param string $email
   *   The user's email address.
   * @param string $provider_user_id
   *   The unique id returned by the user.
   * @param string $token
   *   The access token for making additional API calls.
   * @param string $data
   *   The additional user_data to be stored in database.
   *
   * @return array|\Symfony\Component\HttpKernel\Exception\BadRequestHttpException|bool
   *   True if user could be authenticated with email.
   *   False otherwise.
   */
  public function authenticateWithEmail($email, $provider_user_id, $token, $data) {
    try {
      // Load user by email.
      $drupal_user = $this->userManager->loadUserByProperty('mail', $email);

      // Check if user with same email account exists.
      if ($drupal_user) {
        // Add record for the same user.
        $this->userManager->addUserRecord($drupal_user->id(), $provider_user_id, $token, $data);

        // Authenticates and redirect the user.
        return $this->authenticateExistingUser($drupal_user);
      }
    }
    catch (\Exception $ex) {
      throw new BadRequestHttpException($this->t('Failed to authenticate user. Exception: @message', ['@message' => $ex->getMessage()]));
    }

    return FALSE;
  }

  /**
   * Logs the user in.
   *
   * @param \Drupal\user\UserInterface $drupal_user
   *   User object.
   *
   * @return array|\Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Response info.
   */
  public function loginUser(UserInterface $drupal_user) {
    // Check that the account is active and log the user in.
    if ($drupal_user->isActive()) {
      $this->userLoginFinalize($drupal_user);

      // Send basic metadata about the logged in user.
      $response_data = [];
      if ($drupal_user->get('uid')->access('view', $drupal_user)) {
        $response_data['current_user']['uid'] = $drupal_user->id();
      }
      if ($drupal_user->get('roles')->access('view', $drupal_user)) {
        $response_data['current_user']['roles'] = $drupal_user->getRoles();
      }
      if ($drupal_user->get('name')->access('view', $drupal_user)) {
        $response_data['current_user']['name'] = $drupal_user->getAccountName();
      }
      $response_data['csrf_token'] = $this->csrfToken->get('rest');

      $logout_route = $this->routeProvider->getRouteByName('user.logout.http');
      // Trim '/' off path to match \Drupal\Core\Access\CsrfAccessCheck.
      $logout_path = ltrim($logout_route->getPath(), '/');
      $response_data['logout_token'] = $this->csrfToken->get($logout_path);

      return $response_data;
    }
    throw new BadRequestHttpException($this->t('Login for user @user prevented. Account is blocked.', ['@user' => $drupal_user->getAccountName()]));
  }

}
