<?php

namespace Drupal\social_auth_decoupled;

use Drupal\Component\Plugin\Exception\PluginException;
use League\OAuth2\Client\Token\AccessToken;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Provides a trait for social_auth_decoupled.
 */
trait SocialAuthDecoupledTrait {

  /**
   * The Social Auth user authenticator..
   *
   * @var \Drupal\social_auth_decoupled\User\UserAuthenticatorHttp
   */
  protected $userAuthenticatorHttp;

  /**
   * The serializer.
   *
   * @var \Symfony\Component\Serializer\Serializer
   */
  protected $serializer;

  /**
   * Inject the service serializer.
   */
  public function serializer() {
    if (!isset($this->serializer)) {
      $this->serializer = \Drupal::getContainer()
        ->get('serializer');
    }
    return $this->serializer;
  }

  /**
   * Inject the service social_auth_decoupled.user_authenticator_http.
   */
  public function userAuthenticatorHttp() {
    if (!isset($this->userAuthenticatorHttp)) {
      $this->userAuthenticatorHttp = \Drupal::getContainer()
        ->get('social_auth_decoupled.user_authenticator_http');
    }
    return $this->userAuthenticatorHttp;
  }

  /**
   * Process implementer callback path.
   *
   * @param string $access_token
   *   The access token.
   *
   * @return \League\OAuth2\Client\Provider\GenericResourceOwner|null
   *   The user info if successful.
   *   Null otherwise.
   */
  public function loginProcessCallback($access_token) {
    try {
      /** @var \League\OAuth2\Client\Provider\AbstractProvider|false $client */
      $client = $this->networkManager->createInstance($this->pluginId)
        ->getSdk();

      // If provider client could not be obtained.
      if (!$client) {
        throw new BadRequestHttpException('Social auth not configured properly. Contact site administrator.');
      }

      $this->providerManager->setClient($client);

      $accessToken = new AccessToken([
        'access_token' => $access_token,
        'expires_in' => 3600,
      ]);
      $this->providerManager->setAccessToken($accessToken);
      // Saves access token to session.
      $this->dataHandler->set('access_token', $this->providerManager->getAccessToken());

      // Gets user's info from provider.
      if (!$profile = $this->providerManager->getUserInfo()) {
        throw new BadRequestHttpException('Login failed, could not load user profile. Contact site administrator.');
      }

      return $profile;

    }
    catch (PluginException $exception) {
      throw new BadRequestHttpException('There has been an error when creating plugin.');
    }
  }

  /**
   * Logs in a user.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A response which contains the ID and CSRF token.
   */
  public function login(Request $request) {
    $content = $request->getContent();
    $credentials = $this->serializer()->decode($content, 'json');
    if (!isset($credentials['access_token'])) {
      throw new BadRequestHttpException('Missing access.');
    }
    $access_token = $credentials['access_token'];

    try {
      /** @var \League\OAuth2\Client\Provider\ResourceOwnerInterface|null $profile */
      $profile = $this->loginProcessCallback($access_token);
    }
    catch (\Exception $e) {
      throw new BadRequestHttpException($e->getMessage());
    }

    // If authentication was successful.
    if ($profile !== NULL) {

      // Gets (or not) extra initial data.
      $data = $this->userAuthenticatorHttp()
        ->checkProviderIsAssociated($profile->getId()) ? NULL : $this->providerManager->getExtraDetails();

      try {
        $response_data = $this->authenticateUserByProfile($profile, $data);
      }
      catch (\Exception $e) {
        throw new BadRequestHttpException($e->getMessage());
      }

      $encoded_response_data = $this->serializer()
        ->encode($response_data, 'json');
      return new Response($encoded_response_data);
    }

  }

}
