<?php

namespace Drupal\social_auth_entra_id\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for handling Microsoft Entra ID.
 *
 * Social authentication redirects and callbacks.
 */
class SocialAuthEntraIdController implements ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * Configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * HTTP client service for making external requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Messenger service for displaying messages.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Logger factory service for logging errors.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a SocialAuthEntraIdController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Configuration factory service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   HTTP client service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   Language manager service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   String translation service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    MessengerInterface $messenger,
    LanguageManagerInterface $language_manager,
    LoggerChannelFactoryInterface $logger_factory,
    TranslationInterface $string_translation,
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->messenger = $messenger;
    $this->languageManager = $language_manager;
    $this->loggerFactory = $logger_factory;
    $this->setStringTranslation($string_translation);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('http_client'),
      $container->get('messenger'),
      $container->get('language_manager'),
      $container->get('logger.factory'),
      $container->get('string_translation')
    );
  }

  /**
   * Redirects the user to the Microsoft Entra ID login page.
   *
   * @return \Drupal\Core\Routing\TrustedRedirectResponse
   *   A trusted redirect response to the Microsoft Entra ID login URL.
   */
  public function redirectToMicrosoft() {
    $config = $this->configFactory->get('social_auth_entra_id.settings');
    $client_id = $config->get('client_id');
    $tenant_id = $config->get('tenant_id');

    if (empty($client_id) || empty($tenant_id)) {
      // Add an alert message for empty configuration.
      $this->messenger->addError($this->t('Empty configuration. Please contact the site administrator.'));
      // Redirect to the user login page with the current language.
      $current_language = $this->languageManager->getCurrentLanguage()->getId();
      return new RedirectResponse(Url::fromRoute('user.login', [], ['language' => $current_language])->toString());
    }

    // Use Url::fromRoute() to generate the redirect URI.
    $redirect_uri = Url::fromRoute('social_auth_entra_id.callback', [], ['absolute' => TRUE])->toString(TRUE);
    $scopes = 'openid profile email';

    // Construct the URL for the Microsoft login.
    $url = "https://login.microsoftonline.com/$tenant_id/oauth2/v2.0/authorize?" . http_build_query([
      'client_id' => $client_id,
      'response_type' => 'code',
      'redirect_uri' => $redirect_uri,
      'scope' => $scopes,
    ]);

    // Redirect the user to Microsoft login using TrustedRedirectResponse.
    return new TrustedRedirectResponse($url);
  }

  /**
   * Handles the callback from Microsoft Entra ID after user authorization.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the front page after processing the callback.
   */
  public function handleMicrosoftCallback(Request $request) {
    $code = $request->query->get('code');

    if ($code) {
      // Retrieve settings from configuration.
      $config = $this->configFactory->get('social_auth_entra_id.settings');
      $client_id = $config->get('client_id');
      $client_secret = $config->get('client_secret');
      $tenant_id = $config->get('tenant_id');
      $redirect_uri = Url::fromRoute('social_auth_entra_id.callback', [], ['absolute' => TRUE])->toString();
      $login_behavior = $config->get('login_behavior');
      // Fetch and sanitize allowed domains.
      $allowed_domains_raw = $config->get('allowed_domains') ?? '';
      $allowed_domains = array_filter(array_map('trim', explode(',', $allowed_domains_raw)));

      try {
        // Exchange the code for an access token.
        $response = $this->httpClient->post("https://login.microsoftonline.com/$tenant_id/oauth2/v2.0/token", [
          'form_params' => [
            'client_id' => $client_id,
            'client_secret' => $client_secret,
            'code' => $code,
            'redirect_uri' => $redirect_uri,
            'grant_type' => 'authorization_code',
          ],
        ]);

        $data = json_decode($response->getBody()->getContents(), TRUE);

        // Check if access_token exists in response.
        if (isset($data['access_token'])) {
          $profile_response = $this->httpClient->get('https://graph.microsoft.com/v1.0/me', [
            'headers' => ['Authorization' => 'Bearer ' . $data['access_token']],
          ]);

          $profile_data = json_decode($profile_response->getBody()->getContents(), TRUE);

          if (isset($profile_data['mail'])) {
            $user_email = $profile_data['mail'];
            $user_email_domain = substr(strrchr($user_email, "@"), 1);

            if (!empty($allowed_domains) && !in_array($user_email_domain, $allowed_domains)) {
              $this->messenger->addError($this->t('Your email domain is not allowed.'));

              return new RedirectResponse(Url::fromRoute('<front>')->toString());
            }

            $existing_user = user_load_by_mail($user_email);

            if (!$existing_user && $login_behavior == 'register_and_login') {
              $new_user = User::create([
                'name' => $profile_data['displayName'] ?? $user_email,
                'mail' => $user_email,
                'status' => 1,
              ]);
              $new_user->save();
              user_login_finalize($new_user);
              $this->messenger->addStatus($this->t('Account created and logged in.'));
            }
            elseif ($existing_user) {
              user_login_finalize($existing_user);
              $this->messenger->addStatus($this->t('Logged in successfully.'));
            }
            else {
              $this->messenger->addError($this->t('Login failed. The account does not exist.'));
              return new RedirectResponse(Url::fromRoute('<front>')->toString());
            }
          }
          else {
            throw new \Exception('User email not found.');
          }
        }
        else {
          throw new \Exception('Access token missing.');
        }
      }
      catch (\Exception $e) {
        $this->messenger->addError($this->t('An error occurred.'));
        $this->loggerFactory
          ->get('social_auth_entra_id')
          ->error('Microsoft Entra callback error: @message', ['@message' => $e->getMessage()]);
      }
    }
    else {
      $this->messenger->addError($this->t('Authorization code missing.'));
    }

    // Redirect to the front page after handling callback or errors.
    return new RedirectResponse(Url::fromRoute('<front>')->toString());
  }

}
