<?php

namespace Drupal\social_auth_entra_id\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configures settings for the Entra ID social authentication module.
 *
 * Provides settings for configuring the Entra ID client ID, client secret,
 * and other necessary configurations for Microsoft Entra ID integration.
 */
class SocialAuthEntraIdSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['social_auth_entra_id.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'social_auth_entra_id_settings_form';
  }

  /**
   * Builds the settings form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('social_auth_entra_id.settings');

    // Client ID field.
    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $config->get('client_id'),
      '#description' => $this->t('Enter the Client ID provided by Azure AD.'),
      '#required' => TRUE,
    ];

    // Client Secret field.
    $form['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret'),
      '#default_value' => $config->get('client_secret'),
      '#description' => $this->t('Enter the Client Secret provided by Azure AD.'),
      '#required' => TRUE,
    ];

    // Tenant ID field.
    $form['tenant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tenant ID'),
      '#default_value' => $config->get('tenant_id'),
      '#description' => $this->t('Enter the Tenant ID for your Azure AD application.'),
      '#required' => TRUE,
    ];

    // Login behavior selection.
    $form['login_behavior'] = [
      '#type' => 'radios',
      '#title' => $this->t('Login Behavior'),
      '#options' => [
        'register_and_login' => $this->t('Register and Login'),
        'login_only' => $this->t('Login Only'),
      ],
      '#default_value' => $config->get('login_behavior') ?? 'register_and_login',
      '#description' => $this->t('Choose whether to allow registration and login, or restrict to login only for existing accounts.'),
    ];

    // Domain allowlist field.
    $form['allowed_domains'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed Domains'),
      '#default_value' => $config->get('allowed_domains'),
      '#description' => $this->t('Enter a comma-separated list of email domains that are allowed to register/login. Leave empty to allow all domains. Example: example.com, another-domain.com'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('social_auth_entra_id.settings')
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('client_secret', $form_state->getValue('client_secret'))
      ->set('tenant_id', $form_state->getValue('tenant_id'))
      ->set('login_behavior', $form_state->getValue('login_behavior'))
      ->set('allowed_domains', $form_state->getValue('allowed_domains'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
