<?php

declare(strict_types=1);

namespace Drupal\social_auth_entra_id\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides an Entra ID login block.
 *
 * @Block(
 *   id = "entra_id_login_block",
 *   admin_label = @Translation("Entra ID Login Block"),
 *   category = @Translation("Login"),
 * )
 */
final class EntraIdLoginBlockBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function build(): array {

    // Retrieve the configured login text or use default.
    $login_text = $this->configuration['login_text'] ?? '<i class="fa-brands fa-microsoft"></i> Log in with Microsoft Entra ID';

    // Retrieve the configured custom class or set to a default.
    $custom_class = !empty($this->configuration['custom_class']) ? $this->configuration['custom_class'] : 'btn btn-primary';

    return [
      'content' => [
        '#type' => 'link',
        '#title' => [
          '#markup' => $login_text,
          '#allowed_tags' => ['i', 'strong', 'em', 'b', 'u', 'span'],
        ],
        '#url' => Url::fromRoute('social_auth_entra_id.redirect'),
        '#attributes' => ['class' => [$custom_class]],
        '#attached' => [
          'library' => [
            'social_auth_entra_id/font-awesome',
          ],
        ],
        '#cache' => [
          'contexts' => ['user'],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    // Get the existing configuration.
    $config = $this->getConfiguration();
    $default_class = 'btn btn-primary';

    // Create the form field for the login text.
    $form['login_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Login Text'),
      '#description' => $this->t('Enter the HTML to be displayed for the login link, e.g., <code>&lt;i class="fa-brands fa-microsoft"&gt;&lt;/i&gt; Sign in with Microsoft</code>. Make sure font-awesome is installed for font-awesome icons'),
      '#default_value' => $config['login_text'] ?? '<i class="fa-brands fa-microsoft"></i> Log in with Microsoft Entra ID',
    ];

    // Create the form field for the custom class.
    $form['custom_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Class'),
      '#description' => $this->t('Enter a custom CSS class for the login link.'),
      '#default_value' => $config['custom_class'] ?? $default_class,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    // Save the configuration.
    $this->configuration['login_text'] = $form_state->getValue('login_text');
    $this->configuration['custom_class'] = $form_state->getValue('custom_class');
  }

}
