((Drupal) => {
  /**
   * Google One Tap behavior.
   *
   * Attaches the Google One Tap functionality to the page.
   */
  Drupal.behaviors.googleOneTap = {
    /**
     * Attaches the Google One Tap functionality to the page.
     *
     * @param {HTMLElement} context
     *   The context for which the behavior is being attached.
     * @param {object} settings
     *   The Drupal settings object.
     */
    attach(context, settings) {
      // Only handle the document context (no other AJAX contexts):
      if (context === document) {
        // Get the Google Client ID from drupalSettings.
        const googleClientId = settings.social_auth_google_one_tap?.client_id;
        if (!googleClientId) {
          throw new Error('Google Client ID not found for One Tap.');
        }

        /**
         * Google GSI callback function.
         * This function is globally exposed as required by the Google Identity Services library.
         *
         * @param {object} response
         *   The credential response from Google.
         * @param {string} response.credential
         *   The ID Token (JWT).
         */
        window.handleCredentialResponse = (response) => {
          const idToken = response.credential;

          // Send the ID Token to our backend for validation and login.
          fetch('/user/login/google/one-tap-callback', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({ token: idToken }),
          })
            .then((res) => res.json())
            .then((data) => {
              if (data.logged_in) {
                // On successful login, reload the page to reflect the new session.
                window.location.reload();
              } else {
                // Handle login failure.
                throw new Error('One Tap login failed:', data.message);
              }
            })
            .catch((error) => {
              throw new Error('Error during One Tap callback:', error);
            });
        };

        // Initialize Google One Tap.
        // Check for the 'no_one_tap_auto' cookie.
        const cookies = document.cookie.split('; ');
        const noAutoSelectCookie = cookies.find((row) =>
          row.startsWith('no_one_tap_auto='),
        );
        const autoSelect = !noAutoSelectCookie;

        window.google.accounts.id.initialize({
          client_id: googleClientId,
          callback: window.handleCredentialResponse,
          auto_select: autoSelect,
        });

        // Prompt for One Tap login for non-logged-in users.
        window.google.accounts.id.prompt();
      }
    },
  };
})(Drupal);
