<?php

namespace Drupal\social_auth_google_one_tap\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Drupal\social_auth\User\SocialAuthUser;
use Drupal\social_auth\User\UserAuthenticator;
use Drupal\social_auth\SocialAuthDataHandler;
use Drupal\social_api\User\UserManagerInterface;
use Drupal\social_auth\Event\LoginEvent;
use Drupal\social_auth\Event\SocialAuthEvents;
use Drupal\social_auth\User\UserManager;
use Google\Client as GoogleClient;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Handles the Google One Tap callback.
 */
class OneTapController extends ControllerBase {

  /**
   * The user authenticator service.
   *
   * @var \Drupal\social_auth\User\UserAuthenticator
   */
  protected UserAuthenticator $userAuthenticator;

  /**
   * The social auth data handler service.
   *
   * @var \Drupal\social_auth\SocialAuthDataHandler
   */
  protected SocialAuthDataHandler $dataHandler;

  /**
   * The user manager service.
   * 
   * @todo We should use an interface here, but unfortunately UserManager
   * doesn't implement an interface. Only its parent class, but that interface
   * is missing some methods we are calling here.
   *
   * @var \Drupal\social_auth\User\UserManager
   */
  protected UserManager $userManager;

  /**
   * The event dispatcher service.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected EventDispatcherInterface $dispatcher;

  /**
   * Constructs a new OneTapController object.
   *
   * @param \Drupal\social_auth\User\UserAuthenticator $userAuthenticator
   *   The user authenticator service.
   * @param \Drupal\social_auth\SocialAuthDataHandler $dataHandler
   *   The social auth data handler service.
   * @param \Drupal\social_api\User\UserManagerInterface $userManager
   *   The user manager service.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $dispatcher
   *   The event dispatcher service.
   */
  public function __construct(UserAuthenticator $userAuthenticator, SocialAuthDataHandler $dataHandler, UserManagerInterface $userManager, EventDispatcherInterface $dispatcher) {
    $this->userAuthenticator = $userAuthenticator;
    $this->dataHandler = $dataHandler;
    $this->userManager = $userManager;
    $this->dispatcher = $dispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('social_auth.user_authenticator'),
      $container->get('social_auth.data_handler'),
      $container->get('social_auth.user_manager'),
      $container->get('event_dispatcher'),
    );
  }

  /**
   * Handles the Google One Tap callback.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function callback(Request $request) {
    $content = Json::decode($request->getContent());
    $idToken = $content['token'] ?? NULL;

    if (!$idToken) {
      return new JsonResponse(['logged_in' => FALSE, 'message' => 'Token not provided.'], 400);
    }

    // Get the Client ID from the social_auth_google settings.
    $googleSettings = $this->config('social_auth_google.settings');
    $clientId = $googleSettings->get('client_id');

    if (!$clientId) {
      return new JsonResponse(['logged_in' => FALSE, 'message' => 'Google Client ID is not configured.'], 500);
    }

    try {
      $client = new GoogleClient(['client_id' => $clientId]);
      $payload = $client->verifyIdToken($idToken);

      if ($payload) {
        // Create a SocialAuthUser object to ensure data consistency:
        $socialUser = new SocialAuthUser(
          $payload['name'] ?? '',
          // The unique Google User ID:
          $payload['sub'],
          // We can store the ID token itself as the access token:
          $idToken,
          $payload['email'],
          $payload['picture'] ?? NULL,
          // Extra data:
          []
        );
        $socialUser->setFirstName($payload['given_name'] ?? '');
        $socialUser->setLastName($payload['family_name'] ?? '');

        // Set the plugin ID and save the access token to the data handler
        // BEFORE calling the authenticator methods.
        $this->dataHandler->setSessionPrefix('social_auth_google');
        $this->dataHandler->set('access_token', $idToken);

        // The UserAuthenticator will now be able to retrieve the token
        // internally:
        $this->userAuthenticator->setPluginId('social_auth_google');

        // Check if user already exists.
        $user_id = $this->userAuthenticator->checkProviderIsAssociated($socialUser->getProviderId());

        $drupalUser = NULL;
        if ($user_id) {
          // Load existing user.
          $drupalUser = $this->userManager->loadUserByProperty('uid', $user_id);
        }
        else {
          // Create new user if not found.
          $drupalUser = $this->userManager->createNewUser($socialUser);
        }

        if ($drupalUser) {
          // Log the user in.
          $this->userAuthenticator->loginUser($drupalUser);

          if ($this->currentUser()->isAuthenticated()) {
            $event = new LoginEvent($this->currentUser(), $socialUser, 'social_auth_google');
            $this->dispatcher->dispatch($event, SocialAuthEvents::USER_LOGIN);
          }

          return new JsonResponse(['logged_in' => TRUE]);
        }
        else {
          return new JsonResponse(['logged_in' => FALSE, 'message' => 'Could not find or create user.'], 500);
        }

      }
      else {
        return new JsonResponse(['logged_in' => FALSE, 'message' => 'Invalid ID token.'], 401);
      }
    }
    catch (\Exception $e) {
      $this->getLogger('social_auth_google_one_tap')->error('One Tap login failed: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['logged_in' => FALSE, 'message' => 'An error occurred during validation.'], 500);
    }
  }

}
