<?php

namespace Drupal\social_auth_google_one_tap\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Handles logout cookie.
 */
class LogoutCookieSubscriber implements EventSubscriberInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs a new LogoutCookieSubscriber object.
   *
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(AccountInterface $current_user, RequestStack $request_stack) {
    $this->currentUser = $current_user;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events = [];
    $events[KernelEvents::RESPONSE][] = ['onResponse'];
    return $events;
  }

  /**
   * Handles logout cookie.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The event to process.
   */
  public function onResponse(ResponseEvent $event) {
    // Check if the session flag for logout is set.
    $request = $this->requestStack->getCurrentRequest();
    $session = $request->getSession();

    if ($session->has('social_auth_google_one_tap_logged_out') && $session->get('social_auth_google_one_tap_logged_out') === TRUE) {
      $response = $event->getResponse();
      // Set the cookie to prevent auto-select for 12 hours.
      $cookie_name = 'no_one_tap_auto';
      $cookie_value = '1';
      // 12 hours from now.
      $expire = time() + (12 * 3600);
      $domain = $request->getHost();

      // Create a cookie that is readable by JavaScript (HttpOnly = false).
      $cookie = new Cookie(
        $cookie_name,
        $cookie_value,
        $expire,
        // Path:
        '/',
        // Domain:
        $domain,
        // Secure:
        $request->isSecure(),
        // HttpOnly = false:
        FALSE,
        // Raw:
        FALSE,
        // SameSite:
        Cookie::SAMESITE_LAX,
      );
      $response->headers->setCookie($cookie);

      // Clear the session flag immediately after setting the cookie.
      $session->remove('social_auth_google_one_tap_logged_out');
    }
  }

}
