<?php

namespace Drupal\social_auth_modal\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\social_api\Plugin\NetworkManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Social Auth Block for Modal Login.
 *
 * @Block(
 *   id = "social_auth_login_modal_block",
 *   admin_label = @Translation("Social Auth Modal Login"),
 * )
 */
class SocialAuthModalLoginBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The network manager.
   */
  private NetworkManager $networkManager;

  /**
   * Immutable configuration for social_auth_modal.settings.
   */
  protected ImmutableConfig $socialAuthModalConfig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->socialAuthModalConfig = $container->get('config.factory')
      ->get('social_auth_modal.settings');
    $instance->networkManager = $container->get('plugin.network.manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    // Recent versions of the Social Auth module allow achieving the same
    // results with the default block (SocialAuthLoginBlock) by adjusting the
    // block template. But for the sake of upgrade compatibility, we're keeping
    // the block output structure, which is now different from in the default
    // block.
    $destination = Url::fromRoute('social_auth_modal.complete')->toString();
    $options = [
      'attributes' => [
        'class' => ['social-auth', 'auth-link', 'social-auth-modal__link'],
      ],
      'query' => [
        'destination' => $destination,
      ],
    ];

    $social_networks = [];
    $links = [];

    foreach ($this->networkManager->getDefinitions() as $definition) {
      $network_plugin = $this->networkManager->createInstance($definition['id']);
      $name = $network_plugin->getSocialNetwork() ?? $this->t('Untitled service');
      $url = $network_plugin->getRedirectUrl($options);
      $social_network = [
        'name' => $name,
        'url' => $url,
        'img_path' => $network_plugin->getProviderLogoPath(),
      ];
      $social_networks[$definition['id']] = $social_network;
      $links[$definition['id']] = Link::fromTextAndUrl($name, $url)->toRenderable();
    }

    $modal_config = $this->socialAuthModalConfig;

    return [
      '#attached' => [
        'library' => [
          'social_auth_modal/modal_open',
        ],
        'drupalSettings' => [
          'socialAuthModal' => [
            'width' => $modal_config->get('modal_width'),
            'height' => $modal_config->get('modal_height'),
          ],
        ],
      ],
      // Block content.
      'links' => [
        '#theme' => 'item_list',
        '#items' => $links,
        '#title' => $this->t('Authenticate with'),
      ],
      // Raw data.
      '#social_networks' => $social_networks,
      '#destination' => $destination,
    ];
  }

}
