<?php

namespace Drupal\social_post_instagram\Controller;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\social_api\Plugin\NetworkManager;
use Drupal\social_post\DataHandler;
use Drupal\social_post\Controller\ControllerBase;
use Drupal\social_post\Entity\Controller\SocialPostListBuilder;
use Drupal\social_post\User\UserManager;
use Drupal\social_post_instagram\InstagramPostAuthManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Returns responses for Social Post Instagram routes.
 */
class InstagramPostController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The network plugin manager.
   *
   * @var \Drupal\social_api\Plugin\NetworkManager
   */
  private $networkManager;

  /**
   * The Instagram authentication manager.
   *
   * @var \Drupal\social_post_instagram\InstagramPostAuthManager
   */
  private $instagramManager;

  /**
   * Used to access GET parameters.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private $request;

  /**
   * The Social Auth Data Handler.
   *
   * @var \Drupal\social_post\DataHandler
   */
  private $dataHandler;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The social post user manager.
   *
   * @var \Drupal\social_post\User\UserManager
   */
  protected $userManager;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * InstagramAuthController constructor.
   *
   * @param \Drupal\social_api\Plugin\NetworkManager $network_manager
   *   Used to get an instance of social_auth_instagram network plugin.
   * @param \Drupal\social_post\User\UserManager $user_manager
   *   Manages user login/registration.
   * @param \Drupal\social_post_instagram\InstagramPostAuthManager $instagram_manager
   *   Used to manage authentication methods.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request
   *   Used to access GET parameters.
   * @param \Drupal\social_post\DataHandler $data_handler
   *   The social post data handler.
   * @param \Drupal\social_post\Entity\Controller\SocialPostListBuilder $list_builder
   *   The Social Post entity list builder.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Used for logging errors.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */

  public function __construct(
    NetworkManager $network_manager,
    UserManager $user_manager,
    InstagramPostAuthManager $instagram_manager,
    RequestStack $request,
    DataHandler $data_handler,
    SocialPostListBuilder $list_builder,
    LoggerChannelFactoryInterface $logger_factory,
    MessengerInterface $messenger,
  ) {

    $this->networkManager = $network_manager;
    $this->userManager = $user_manager;
    $this->instagramManager = $instagram_manager;
    $this->request = $request;
    $this->dataHandler = $data_handler;
    $this->listBuilder = $list_builder;
    $this->loggerFactory = $logger_factory;
    $this->messenger = $messenger;

    $this->userManager->setPluginId('social_post_instagram');

    // Sets session prefix for data handler.
    $this->dataHandler->getSessionPrefix('social_post_instagram');
  }

  /**
   * {@inheritdoc}
   * @param ContainerInterface $container
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.network.manager'),
      $container->get('social_post.user_manager'),
      $container->get('instagram_post.social_post_auth_manager'),
      $container->get('request_stack'),
      $container->get('social_post.data_handler'),
      $container->get('entity_type.manager')->getListBuilder('social_post'),
      $container->get('logger.factory'),
      $container->get('messenger'),
    );
  }

  /**
   * Redirects the user to FB for authentication of sitewide account.
   */
  public function redirectToIGForSitewide() {
    $this->dataHandler->set('user_id', 0);
    return $this->redirectToIG();
  }

  /**
   * Redirects the user to FB for authentication of user account.
   */
  public function redirectToIGForUser() {
    $this->dataHandler->set('user_id', $this->currentUser()->id());
    return $this->redirectToIG();
  }

  /**
   * Redirects the user to FB for authentication.
   */
  public function redirectToIG() {
    /** @var \Agaric\FacebookApi\Facebook $instagram */
    $instagram = $this->networkManager->createInstance('social_post_instagram')->getSdk();

    // If instagram client could not be obtained.
    if (!$instagram) {
      $this->messenger()->addError($this->t('Social Post Instagram not configured properly. Contact site administrator.'));
      return $this->redirect('user.login');
    }

    // Instagram service was returned, inject it to $instagramManager.
    $this->instagramManager->setClient($instagram);

    // Generates the URL where the user will be redirected for FB login.
    // If the user did not have email permission granted on previous attempt,
    // we use the re-request URL requesting only the email address.
    $fb_login_url = $this->instagramManager->getAuthorizationUrl();

    $state = $this->instagramManager->getState();

    $this->dataHandler->set('oAuth2State', $state);

    return new TrustedRedirectResponse($fb_login_url);
  }

  /**
   * Response for path 'user/login/instagram/callback'.
   *
   * Instagram returns the user here after user has authenticated in FB.
   */
  public function callback() {
    // Checks if user cancel login via Instagram.
    $error = $this->request->getCurrentRequest()->get('error');

    $user_id = $this->dataHandler->get('user_id');

    if ($error == 'access_denied') {
      $this->messenger()->addError($this->t('You could not be authenticated.'));
      return $this->redirect('entity.user.edit_form', ['user' => $this->currentUser()->id()]);
    }

    /** @var \League\OAuth2\Client\Provider\Facebook false $instagram */
    $instagram = $this->networkManager->createInstance('social_post_instagram')->getSdk();
    $this->instagramManager->setClient($instagram);

    // If instagram client could not be obtained.
    if (!$instagram) {
      $this->messenger()->addError($this->t('Social Post Instagram not configured properly. Contact site administrator.'));
      return $this->redirect('user.login');
    }

    $state = $this->dataHandler->get('oAuth2State');

    // Retrieves $_GET['state'].
    $retrievedState = $this->request->getCurrentRequest()->query->get('state');
    if (empty($retrievedState) || ($retrievedState !== $state)) {
      $this->messenger()->addError($this->t('Instagram login failed. Invalid oAuth2 State.'));
      return $this->redirect('user.login');
    }

    $this->instagramManager->authenticate();

    // User's instagram profile info.
    $userInfo = $this->instagramManager->getUserInfo();
    $profile = $userInfo['user'];
    $pages = $userInfo['pages'];

    if ($this->userManager->getDrupalUserId($profile->getId()) !== FALSE) {
      $this->messenger->addWarning($this->t('You have already authorized posting on behalf of this account.'));
    }
    elseif (count($pages) > 1) {
      $this->messenger->addWarning($this->t('Unable to add account because more than one page was selected during authorization. Please try again and make sure to "Edit previous settings" during authorization to resolve the error.'));
    }
    else {
      $this->userManager->addUserRecord($profile->getName(), $user_id, $profile->getId(), $profile->getLink(), $this->instagramManager->getAccessToken(), serialize($pages));
      $this->messenger->addStatus($this->t('Account added successfully.'));
    }

    if ($user_id == 0) {
      return $this->redirect('social_post_instagram.sitewide_account_form');
    } else {
      return $this->redirect('entity.user.edit_form', ['user' => $user_id]);
    }
  }

}
