<?php

namespace Drupal\social_post_instagram;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\social_api\Plugin\NetworkManager;
use Drupal\social_api\SocialApiException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\Session;
use Drupal\social_post\PostManager\OAuth2Manager;

/**
 * Manages the authorization process and post on user behalf.
 */
class InstagramPostAuthManager extends OAuth2Manager {
  /**
   * The session manager.
   *
   * @var \Symfony\Component\HttpFoundation\Session\Session
   */
  protected $session;

  /**
   * The Facebook client object.
   *
   * @var \League\OAuth2\Client\Provider\Facebook
   */
  protected mixed $client;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected ?Request $request;

  /**
   * The Social Api network manager.
   *
   * @var \Drupal\social_api\Plugin\NetworkManager
   */
  protected $networkManager;

  /**
   * The Facebook access token.
   *
   * @var \League\OAuth2\Client\Token\AccessToken
   */
  protected $token;

  /**
   * The Facebook page long lived access token.
   *
   * @var string
   */
  protected $longLivedToken;

  /**
   * The post data.
   *
   * @var array
   */
  protected $postData;

  /**
   * The target pageId for posting.
   *
   * @var array
   */
  protected $pageId;

  /**
   * InstagramPostAuthManager constructor.
   *
   * @param \Symfony\Component\HttpFoundation\Session\Session $session
   *   The session manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Used to get the parameter code returned by Facebook.
   * @param \Drupal\social_api\Plugin\NetworkManager $network_manager
   *   The Social API network manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(Session $session, RequestStack $request_stack, NetworkManager $network_manager, ConfigFactoryInterface $configFactory) {
    $this->session = $session;
    $this->request = $request_stack->getCurrentRequest();
    $this->networkManager = $network_manager;
    $this->settings = $configFactory->get('social_post_instagram.settings');
  }

  /**
   * Saves access token.
   */
  public function authenticate() {
    $this->token = $this->client->getAccessToken(
      'authorization_code',
      ['code' => $this->request->query->get('code')]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthorizationUrl(): mixed {
    $options = ['scopes' => ['email']];

    if ($config_id = $this->settings->get('config_id')) {
      $options = ['config_id' => $config_id];
    }
    return $this->client->getAuthorizationUrl($options);
  }

  /**
   * Gets the data by using the access token returned.
   *
   * @return array
   *   User Info returned by the facebook.
   */
  public function getUserInfo(): array {
    $this->user = $this->client->getResourceOwner($this->token);
    return [
      'user' => $this->user,
      'pages' => $this->client->getAllLongLivedPageAccessTokens($this->token),
    ];
  }

  /**
   * Returns token generated after authorization.
   *
   * @return string
   *   Used for making API calls.
   */
  public function getAccessToken(): mixed {
    return $this->token;
  }

  /**
   * Returns the OAuth login URL where user will be redirected.
   *
   * @return string
   *   Absolute OAuth login URL where user will be redirected
   */
  public function getState(): string {
    return $this->client->getState();
  }
}
