<?php

namespace Drupal\social_post_instagram\Plugin\Action;

use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\media\MediaInterface;
use Drupal\social_post_instagram\Services\InstagramApiService;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
/**
 * Post media to Instagram.
 *
 * @Action(
 *   id = "social_post_instagram_post_media",
 *   label = @Translation("Post to Instagram"),
 *   type = "media"
 * )
 */
class PostToInstagram extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * The Instagram API service.
   *
   * @var \Drupal\social_post_instagram\Services\InstagramApiService
   */
  protected $instagramApiService;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a PostToInstagram object.
   */
    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

  public function __construct(
    array $configuration,
          $plugin_id,
          $plugin_definition,
    InstagramApiService $instagram_api_service,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->instagramApiService = $instagram_api_service;
    $this->loggerFactory = $logger_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('social_post_instagram.instagram_api'),
      $container->get('logger.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array {
    return [
        'caption' => '',
        'tag' => []
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $terms = $this->entityTypeManager->getStorage('taxonomy_term')->loadByProperties(['vid' => 'tags']);

    $form['caption'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Caption'),
      '#description' => $this->t('Optional caption override for Instagram post.'),
      '#default_value' => $this->configuration['caption'] ?? '',
    ];
    $form['tag'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'taxonomy_term',
      '#tags' => TRUE,
      '#selection_handler' => 'default',
      '#selection_settings' => [
        'target_bundles' => ['tags'],
      ],
      '#default_value'=>$terms
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['caption'] = $form_state->getValue('caption');
    $this->configuration['tag'] = $form_state->getValue('tag');
  }

  /**
   *Retrieves the labels of all tags associated with this configuration.
   *
   * @return array
   */
  public function getTags(): array
  {
    $tag_titles=[];
    foreach($this->configuration['tag'] as $tag){
      $tag = Term::load($tag['target_id']);
      $tag_titles[] = $tag->label();
    }
    return $tag_titles;
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    // checking that the entity is a media entity
    if (!$entity instanceof MediaInterface) {
      return;
    }

    try {
      $tags = $this->getTags();
      $tagTitles = implode(' ', $tags);
      $caption = $this->configuration['caption']."\n".$tagTitles;
      // Media Container erstellen
      $media_id = $this->instagramApiService->createMediaContainer($entity, $caption);

      // Media veröffentlichen
      $this->instagramApiService->publishMedia($media_id);

      $this->loggerFactory->get('social_post_instagram')->notice(
        'Media entity @id has been posted to Instagram.',
        ['@id' => $entity->id()]
      );

    } catch (\Exception $e) {
      $this->loggerFactory->get('social_post_instagram')->error(
        'Error posting media @id to Instagram: @error',
        [
          '@id' => $entity->id(),
          '@error' => $e->getMessage(),
        ]
      );
    }
  }
  public function executeMultiple(array $entities) {
    if (count($entities) <= 1) {
      $entity = $entities[0];
      $this->execute($entity);
    }else{
      foreach($entities as $entity) {
        $media_id = $this->instagramApiService->createMediaContainer($entity, '');
        $results[] = $media_id;

      }
      if (!empty($results)) {
        $tags = $this->getTags();
        $tagTitles = implode(' ', $tags);
        $caption = $this->configuration['caption']."\n".$tagTitles;

        $media_ids = implode(',', $results);

        $carousel_id = $this->instagramApiService->createCarouselContainer($media_ids, $caption);

        $this->instagramApiService->publishMedia($carousel_id);

        $this->loggerFactory->get('social_post_instagram')->notice('Media entity with ID has been created and posted to Instagram.');
      }
    }

}

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    /** @var \Drupal\media\MediaInterface $object */
    $result = $object->access('update', $account, TRUE);
    return $return_as_object ? $result : $result->isAllowed();
  }

}
