<?php

namespace Drupal\social_post_instagram\Plugin\Network;

use Agaric\FacebookApi\Facebook;
use Drupal\social_post\Plugin\Network\NetworkBase;
use Drupal\social_api\SocialApiException;
use Drupal\Core\Url;
use Drupal\social_post_instagram\Settings\InstagramPostSettings;



/**
 * Defines Social Post Instagram Network Plugin.
 *
 * @Network(
 *   id = "social_post_instagram",
 *   social_network = "instagram",
 *   type = "social_post",
 *   handlers = {
 *     "settings": {
 *        "class": "\Drupal\social_post_instagram\Settings\InstagramPostSettings",
 *        "config_id": "social_post_instagram.settings"
 *      }
 *   }
 * )
 */
class InstagramPost extends NetworkBase implements InstagramPostInterface {

  /**
   * {@inheritdoc}
   */
  protected function initSdk(): mixed {
    /** @var \Drupal\social_post_instagram\Settings\InstagramPostSettings $settings */
    $settings = $this->settings;
    if (!$this->validateConfig($settings)) {
      throw new SocialApiException('The Social instagram Configuration is not valid');
    }

    return new Facebook([
      'clientId'          => $settings->getAppId(),
      'clientSecret'      => $settings->getAppSecret(),
      'redirectUri'       => $settings->getOauthRedirectUrl(),
      'graphApiVersion'   => 'v' . $settings->getGraphVersion() . '.0',
    ]);
  }

  /**
   * Checks that module is configured.
   *
   * @param \Drupal\social_post_instagram\Settings\InstagramPostSettings $settings
   *   The instagram auth settings.
   *
   * @return bool
   *   True if module is configured.
   *   False otherwise.
   */
  protected function validateConfig(InstagramPostSettings $settings) {
    $app_id = $settings->getAppId();
    $app_secret = $settings->getAppSecret();
    $graph_version = $settings->getGraphVersion();
    if (!$app_id || !$app_secret || !$graph_version) {
      $this->loggerFactory
        ->get('social_post_instagram')
        ->error('Define App ID and App Secret on module settings.');
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getOauthCallback() {
    return Url::fromRoute('social_post_instagram.callback')->setAbsolute()->toString();
  }

}
