<?php

namespace Drupal\social_post_instagram\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\social_post\Entity\SocialPost;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service-Class for the logic to create and publish Instagram Posts
 */
class InstagramApiService {

    use StringTranslationTrait;

    /**
     * Instagram Graph API URL
     */
    protected const INSTAGRAM_API_URL = 'https://graph.facebook.com/';

    /**
     * @var \GuzzleHttp\ClientInterface
     */
    protected $httpClient;

    /**
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * @var \Drupal\Core\Session\AccountProxyInterface
     */
    protected $account;

    /**
     * The file URL generator service.
     *
     * @var \Drupal\Core\File\FileUrlGeneratorInterface
     */
    protected $fileUrlGenerator;

    /**
     * API version.
     *
     * @var string
     */
    protected $apiVersion = 'v24.0';

    /**
     * Constructor.
     *
     * @param \GuzzleHttp\ClientInterface $http_client
     *   The HTTP client.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Core\Messenger\MessengerInterface $messenger
     *   The messenger service.
     * @param \Drupal\Core\Session\AccountProxyInterface $account
     *   The current user account.
     * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
     *   The file URL generator service.
     */
    public function __construct(
        ClientInterface $http_client,
        ConfigFactoryInterface $config_factory,
        MessengerInterface $messenger,
        AccountProxyInterface $account,
        FileUrlGeneratorInterface $file_url_generator
    ) {
        $this->httpClient = $http_client;
        $this->configFactory = $config_factory;
        $this->messenger = $messenger;
        $this->account = $account;
        $this->fileUrlGenerator = $file_url_generator;
    }

    /**
     * Generates an absolute URL for a media image in 'instagram' image style.
     *
     * Creates a derivative of the media image using the 'instagram' image style
     * and returns the absolute URL to the styled image.
     *
     * @param \Drupal\media\Entity\Media $media
     *   The media entity containing the image.
     *
     * @return string|null
     *   The absolute URL to the styled image, or NULL if not found.
     */
    public function getMediaLink(Media $media) {
        $file_id = $media->getSource()->getSourceFieldValue($media);
        $url = null;

        if ($file_id) {
            $file = File::load($file_id);
            if ($file) {
                $uri = $file->getFileUri();
                $url = $this->fileUrlGenerator->generateAbsoluteString($uri);
            }
        }

        return $url;
    }

  /**
   * Retrieves the Instagram access token for the current user account.
   *
   * Searches through all Social Post entities to find an Instagram access token.
   * Prioritizes tokens associated with the current user account. If no
   * user-specific token is found, returns a global token (user_id = 0) as
   * fallback.
   *
   * @return string|null
   *   The access token if found, NULL otherwise.
   */
  protected function getAccessToken() {
    $socialPosts = SocialPost::loadMultiple();
    $account_id = (string) $this->account->id();
    $fallbackToken = NULL;

    foreach ($socialPosts as $socialPost) {
      if ($socialPost->getPluginId() !== 'social_post_instagram') {
        continue;
      }

      $target_id = (string) $socialPost->get('user_id')->target_id;
      $additional = $socialPost->get('additional_data')->getValue();
      if (empty($additional) || !isset($additional[0]['value'])) {
        continue;
      }

      // Unserialize the additional data to extract the access token.
      $data = @unserialize($additional[0]['value'], ['allowed_classes' => FALSE]);
      if ($data === FALSE) {
        continue;
      }

      // Get the first entry.
      $entry = is_array($data) ? reset($data) : $data;
      if (!is_array($entry) || empty($entry['access_token'])) {
        continue;
      }

      $token = $entry['access_token'];

      // Prefer a token for the current account.
      if ($target_id === $account_id) {
        return $token;
      }
      if ($target_id === '0' && $fallbackToken === NULL) {
        $fallbackToken = $token;
      }
    }

    return $fallbackToken;
  }

  /**
   * Retrieves the Instagram User ID from the Instagram Graph API.
   *
   * Makes an API request to fetch the Instagram business account ID associated
   * with the current access token. Returns the first available account ID.
   *
   * @return string|null
   *   The Instagram User ID if successfully retrieved, NULL on failure.
   */
  public function getInstagramUserId() {
    $accessToken = $this->getAccessToken();
    try {
      $response = $this->httpClient->get(self::INSTAGRAM_API_URL . "/{$this->apiVersion}/me/instagram_accounts", [
        'query' => [
          'access_token' => $accessToken,
        ],
      ]);

      $data = json_decode($response->getBody(), TRUE);
      if (is_array($data) && isset($data['data'][0]['id'])) {
        return $data['data'][0]['id'];
      }
    } catch (GuzzleException $e) {
      $this->messenger->addError($this->t('Failed to get Instagram User ID: @m', ['@m' => $e->getMessage()]));
      return NULL;
    }
  }

  /**
   * Sends a Post request to the Instagram API to create a media container with an image
   *
   * @param Media $image
   *  The image
   * @param string $caption
   *  a caption
   *
   * @return string|null
   *  either the container ID or Null on failure
   */
  public function createMediaContainer(Media $image, ?string $caption = '') {
    $access_token = $this->getAccessToken();
    $user_id = $this->getInstagramUserId();
    $media_url = $this->getMediaLink($image);

    // Post request gets filled with parameters and gets executed
    try {
      $response = $this->httpClient->post(self::INSTAGRAM_API_URL."/{$this->apiVersion}/{$user_id}/media", [
        'form_params' => [
          'image_url' => $media_url,
          'caption' => $caption,
          'access_token' => $access_token,
        ],
      ]);
      // $data contains the response array from which the 'id' is extracted.
      $data = json_decode($response->getBody(), TRUE);
      if (is_array($data) && isset($data['id'])) {
        return $data['id'];
      }
      // Error case: JSON could not be decoded or 'id' is missing.
      $this->messenger->addError($this->t('Invalid response from Instagram API when creating media container.'));
      return NULL;

    } catch (\Exception $e) {
      $this->messenger->addError($this->t('Failed to create media container: @m', ['@m' => $e->getMessage()]));
      return NULL;
    }
  }

  public function createCarouselContainer(string $media_ids, string $caption = '') {
    $access_token = $this->getAccessToken();
    $user_id = $this->getInstagramUserId();

    try {
      $response = $this->httpClient->post(self::INSTAGRAM_API_URL."/{$this->apiVersion}/{$user_id}/media", [
        'form_params' => [
          'caption' => $caption,
          'access_token' => $access_token,
          'media_type' => 'CAROUSEL',
          'children'=> $media_ids,
        ],
      ]);
      // $data contains the response array from which the 'id' is extracted.
      $data = json_decode($response->getBody(), TRUE);
      if (is_array($data) && isset($data['id'])) {
        return $data['id'];
      }
      // Error case: JSON could not be decoded or 'id' is missing.
      $this->messenger->addError($this->t('Invalid response from Instagram API when creating media container.'));
      return NULL;

    } catch (\Exception $e) {
      $this->messenger->addError($this->t('Failed to create media container: @m', ['@m' => $e->getMessage()]));
      return NULL;
    }
  }

  /**
   * Send a post request to publish the media container
   * @param string $media_id
   *    Media container ID
   * @return string|null
   *    The published media ID or NULL on failure
   */
  public function publishMedia(string $media_id) {
    $access_token = $this->getAccessToken();
    $user_id = $this->getInstagramUserId();

    // POST-Request for publishing the media container
    try {
      $response = $this->httpClient->post(self::INSTAGRAM_API_URL."/{$this->apiVersion}/{$user_id}/media_publish", [
        'form_params' => [
          'creation_id' => $media_id,
          'access_token' => $access_token,
        ],
      ]);
      // $data contains the response array from which the 'id' is extracted.
      $data = json_decode($response->getBody(), TRUE);
      if (is_array($data) && isset($data['id'])) {
        return $data['id'];
      }
      return NULL;

    } catch (\Exception $e) {
      $this->messenger->addError($this->t('Failed to publish Instagram post. @msg', ['@msg' => $e->getMessage()]));
      return NULL;
    }
  }

}
