<?php

namespace Drupal\social_post_instagram\Settings;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\social_api\Settings\SettingsBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns the app information.
 */
class InstagramPostSettings extends SettingsBase implements InstagramPostSettingsInterface, ContainerInjectionInterface {

    /**
     * Application ID.
     *
     * @var string
     */
    protected $appId;

    /**
     * Application secret.
     *
     * @var string
     */
    protected $appSecret;

    /**
     * The default graph version.
     *
     * @var string
     */
    protected $graphVersion;

    /**
     * The default access token.
     *
     * @var string
     */
    protected $defaultToken;

    /**
     * The redirect URL for social_auth implementer.
     *
     * @var string
     */
    protected $oauthRedirectUrl;

    /**
     * The key repository service.
     *
     * @var \Drupal\key\KeyRepositoryInterface
     */
    protected KeyRepositoryInterface $keyRepository;

    /**
     * Constructor.
     *
     * @param \Drupal\Core\Config\ImmutableConfig $config
     *   The configuration object.
     * @param \Drupal\key\KeyRepositoryInterface $key_repository
     *   The key repository service.
     */
    public function __construct(ImmutableConfig $config, KeyRepositoryInterface $key_repository) {
        parent::__construct($config);
        $this->keyRepository = $key_repository;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container) {
        return new static(
            $container->get('config.factory')->get('social_post_instagram.settings'),
            $container->get('key.repository')
        );
    }

  /**
   * {@inheritdoc}
   */
  public function getAppId() {
    if (!$this->appId) {
      $this->appId = $this->config->get('app_id');
    }
    return $this->appId;
  }

  /**
   * {@inheritdoc}
   */
  public function getAppSecret() {
    if (!$this->appSecret) {
      $app_secret_key = $this->config->get('app_secret');
      // App Secret is stored in key module, so we need to get the actual value.
      $key = $this->keyRepository->getKey($app_secret_key);
      if ($key) {
        $this->appSecret = $key->getKeyValue();
      }
    }
    return $this->appSecret;
  }

  /**
   * {@inheritdoc}
   */
  public function getGraphVersion() {
    if (!$this->graphVersion) {
      $this->graphVersion = $this->config->get('graph_version');
    }
    return $this->graphVersion;
  }
  /**
   * Retrieves the OAuth redirect URL from configuration.

   * @return string|null
   *   The OAuth redirect URL.
   */
  public function getOauthRedirectUrl() {
    if (!$this->oauthRedirectUrl) {
      $this->oauthRedirectUrl = $this->config->get('oauth_redirect_url');
    }
    return $this->oauthRedirectUrl;
  }

}
