<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Commands;

use Drupal\social_summaries\Utility\PlatformConstraints;
use Drush\Commands\DrushCommands;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\social_summaries\Generator\GenerationManager;
use Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager;
use Drupal\social_summaries\Provider\AiProviderInterface;
use Drupal\social_summaries\Service\RateLimiter;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Drush commands for Social Summaries.
 */
class SocialSummariesCommands extends DrushCommands {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $etm;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The generation manager service.
   *
   * @var \Drupal\social_summaries\Generator\GenerationManager
   */
  protected GenerationManager $manager;

  /**
   * The provider plugin manager.
   *
   * @var \Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager
   */
  protected SocialSummariesProviderManager $providerManager;

  /**
   * The rate limiter service.
   *
   * @var \Drupal\social_summaries\Service\RateLimiter
   */
  protected RateLimiter $rateLimiter;

  /**
   * The active AI provider (overridable for tests).
   *
   * @var \Drupal\social_summaries\Provider\AiProviderInterface|null
   */
  protected ?AiProviderInterface $provider = NULL;

  /**
   * Constructs a SocialSummariesCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $etm
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\social_summaries\Generator\GenerationManager $manager
   *   The generation manager service.
   * @param \Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager $provider_manager
   *   The provider plugin manager.
   * @param \Drupal\social_summaries\Service\RateLimiter $rate_limiter
   *   The rate limiter service.
   */
  public function __construct(EntityTypeManagerInterface $etm, ConfigFactoryInterface $config_factory, GenerationManager $manager, SocialSummariesProviderManager $provider_manager, RateLimiter $rate_limiter) {
    $this->etm = $etm;
    $this->configFactory = $config_factory;
    $this->manager = $manager;
    $this->providerManager = $provider_manager;
    $this->rateLimiter = $rate_limiter;
  }

  /**
   * Generate AI content for a node.
   *
   * @param int $nid
   *   The node ID.
   * @param array $options
   *   Additional options.
   *
   * @option platforms
   *   Comma-separated list of platforms to generate
   *   (summary, linkedin, x, newsletter, facebook, threads, all).
   * @option dry-run
   *   Show what would be generated without actually generating.
   * @option force
   *   Force generation even if content already exists.
   *
   * @command social-summaries:gen
   * @aliases ss-gen
   */
  public function gen($nid, array $options = ['platforms' => 'summary']): void {
    $storage = $this->etm->getStorage('node');
    $node = $storage->load($nid);
    if (!$node) {
      $this->loggerSafe()->error('Node {nid} not found.', ['nid' => $nid]);
      return;
    }

    $title = (string) $node->label();
    $body = $node->hasField('body') ? (string) $node->get('body')->value : '';

    $platforms = explode(',', (string) ($options['platforms'] ?? 'summary'));
    if (in_array('all', $platforms, TRUE)) {
      $platforms = ['summary', 'linkedin', 'x', 'newsletter', 'facebook', 'threads'];
    }

    $config = $this->configFactory->get('social_summaries.settings');
    $provider_id = (string) ($config->get('provider') ?? 'openai');
    $provider = $this->provider ?? $this->providerManager->createInstance($provider_id);

    foreach ($platforms as $p) {
      $messages = $this->manager->buildMessages($title, $body, $p);
      $resp = $provider->generate($messages);
      $text = trim($resp['text'] ?? '');
      $tokens_in = (int) ($resp['usage']['tokens_in'] ?? 0);
      $tokens_out = (int) ($resp['usage']['tokens_out'] ?? 0);
      $model = $resp['model'] ?? 'unknown';

      switch ($p) {
        case 'summary':
          if ($node->hasField('field_ai_summary')) {
            $node->set('field_ai_summary', $text);
          }

          break;

        case 'linkedin':
          if ($node->hasField('field_ai_social_linkedin')) {
            $node->set('field_ai_social_linkedin', $text);
          }

          break;

        case 'x':
          if ($node->hasField('field_ai_social_x')) {
            $node->set('field_ai_social_x', $text);
          }

          break;

        case 'newsletter':
          if ($node->hasField('field_ai_newsletter_blurb')) {
            $node->set('field_ai_newsletter_blurb', $text);
          }

          break;

        case 'facebook':
          if ($node->hasField('field_ai_social_facebook')) {
            $node->set('field_ai_social_facebook', $text);
          }

          break;

        case 'threads':
          if ($node->hasField('field_ai_social_threads')) {
            $node->set('field_ai_social_threads', $text);
          }
          break;
      }

      $this->manager->logUsage((int) $node->id(), $p, $model, $tokens_in, $tokens_out);
      $this->loggerSafe()->notice('Generated {platform} for node {nid}.', ['platform' => $p, 'nid' => $nid]);
    }

    $node->setNewRevision(TRUE);
    $node->save();
    $this->loggerSafe()->notice('Saved new revision with AI content for node {nid}.', ['nid' => $nid]);
  }

  /**
   * Generate AI content for multiple nodes in bulk.
   *
   * @command social-summaries:bulk
   * @aliases ss-bulk
   * @option type
   *   Content type to process (default: article).
   * @option platforms
   *   Comma-separated platforms (summary,linkedin,x,newsletter,all).
   * @option limit
   *   Maximum number of nodes to process (default: 50).
   * @option batch-size
   *   Number of nodes to process in each batch (default: 5).
   * @option delay
   *   Delay between API calls in seconds (default: 1).
   * @option dry-run
   *   Show what would be processed without actually generating content.
   * @usage drush ss:bulk --type=article --platforms=summary,linkedin --limit=20
   */
  public function bulk(
    array $options = [
      'type' => 'article',
      'platforms' => 'summary',
      'limit' => 50,
      'batch-size' => 5,
      'delay' => 1,
      'dry-run' => FALSE,
    ],
  ): void {
    $content_type = $options['type'] ?? 'article';
    $platforms = explode(',', (string) ($options['platforms'] ?? 'summary'));
    $limit = (int) ($options['limit'] ?? 50);
    $batch_size = (int) ($options['batch-size'] ?? 5);
    $delay = (float) ($options['delay'] ?? 1.0);
    $dry_run = $options['dry-run'] ?? FALSE;

    if (in_array('all', $platforms, TRUE)) {
      $platforms = ['summary', 'linkedin', 'x', 'newsletter', 'facebook', 'threads'];
    }

    // Get nodes to process.
    $query = $this->etm->getStorage('node')->getQuery()
      ->condition('type', $content_type)
      ->condition('status', 1)
      ->range(0, $limit)
      ->accessCheck(FALSE);
    $nids = $query->execute();

    if (empty($nids)) {
      $this->loggerSafe()->warning('No published {type} nodes found.', ['type' => $content_type]);
      return;
    }

    $total_nodes = count($nids);
    $total_operations = $total_nodes * count($platforms);

    $this->loggerSafe()->notice('Starting bulk processing: {nodes} nodes, {platforms} platforms, {operations} total operations.', [
      'nodes' => $total_nodes,
      'platforms' => implode(', ', $platforms),
      'operations' => $total_operations,
    ]);

    if ($dry_run) {
      $this->loggerSafe()->info('DRY RUN: Would process {nodes} nodes with platforms: {platforms}', [
        'nodes' => $total_nodes,
        'platforms' => implode(', ', $platforms),
      ]);
      return;
    }

    // Determine provider for generation.
    $config = $this->configFactory->get('social_summaries.settings');
    $provider_id = (string) ($config->get('provider') ?? 'openai');
    $provider = $this->provider ?? $this->providerManager->createInstance($provider_id);

    // Process in batches.
    $processed = 0;
    $errors = 0;
    $start_time = time();

    foreach (array_chunk($nids, $batch_size) as $batch_nids) {
      $this->loggerSafe()->info('Processing batch: {current}/{total} nodes', [
        'current' => $processed + count($batch_nids),
        'total' => $total_nodes,
      ]);

      foreach ($batch_nids as $nid) {
        try {
          $node = $this->etm->getStorage('node')->load($nid);
          if (!$node) {
            $this->loggerSafe()->warning('Node {nid} not found, skipping.', ['nid' => $nid]);
            $errors++;
            continue;
          }

          $title = (string) $node->label();
          $body = $node->hasField('body') ? (string) $node->get('body')->value : '';

          if (empty($title) && empty($body)) {
            $this->loggerSafe()->warning('Node {nid} has no title or body content, skipping.', ['nid' => $nid]);
            $errors++;
            continue;
          }

          $node_processed = FALSE;
          foreach ($platforms as $platform) {
            try {
              // Check rate limits before making API call.
              if (!$this->rateLimiter->canMakeCall()) {
                $this->loggerSafe()->warning('Rate limit reached, waiting before continuing...');
                $this->rateLimiter->waitForRateLimit();
              }

              // Add delay between API calls.
              if ($delay > 0) {
                // Convert seconds to microseconds.
                usleep((int) round($delay * 1000000));
              }

              // Check if we should slow down.
              if ($this->rateLimiter->shouldSlowDown()) {
                $recommended_delay = $this->rateLimiter->getRecommendedDelay();
                $this->loggerSafe()->info('Approaching rate limit, adding {delay} second delay', ['delay' => $recommended_delay]);
                sleep((int) ceil($recommended_delay));
              }

              $messages = $this->manager->buildMessages($title, $body, $platform);
              $resp = $provider->generate($messages);
              $text = trim($resp['text'] ?? '');
              $tokens_in = (int) ($resp['usage']['tokens_in'] ?? 0);
              $tokens_out = (int) ($resp['usage']['tokens_out'] ?? 0);
              $model = $resp['model'] ?? 'unknown';

              // Save to appropriate field.
              switch ($platform) {
                case 'summary':
                  if ($node->hasField('field_ai_summary')) {
                    $node->set('field_ai_summary', $text);
                  }

                  break;

                case 'linkedin':
                  if ($node->hasField('field_ai_social_linkedin')) {
                    $node->set('field_ai_social_linkedin', $text);
                  }

                  break;

                case 'x':
                  if ($node->hasField('field_ai_social_x')) {
                    $node->set('field_ai_social_x', $text);
                  }

                  break;

                case 'newsletter':
                  if ($node->hasField('field_ai_newsletter_blurb')) {
                    $node->set('field_ai_newsletter_blurb', $text);
                  }

                  break;

                case 'facebook':
                  if ($node->hasField('field_ai_social_facebook')) {
                    $node->set('field_ai_social_facebook', $text);
                  }

                  break;

                case 'threads':
                  if ($node->hasField('field_ai_social_threads')) {
                    $node->set('field_ai_social_threads', $text);
                  }

                  break;
              }

              $this->manager->logUsage((int) $node->id(), $platform, $model, $tokens_in, $tokens_out);
              $node_processed = TRUE;
            }
            catch (\Exception $e) {
              $this->loggerSafe()->error('Failed to generate {platform} for node {nid}: {error}', [
                'platform' => $platform,
                'nid' => $nid,
                'error' => $e->getMessage(),
              ]);
              $errors++;
            }
          }

          if ($node_processed) {
            try {
              $node->setNewRevision(TRUE);
              $node->save();
              $processed++;
            }
            catch (\Exception $e) {
              $this->loggerSafe()->error('Failed to save node {nid}: {error}', [
                'nid' => $nid,
                'error' => $e->getMessage(),
              ]);
              $errors++;
            }
          }
        }
        catch (\Exception $e) {
          $this->loggerSafe()->error('Error processing node {nid}: {error}', [
            'nid' => $nid,
            'error' => $e->getMessage(),
          ]);
          $errors++;
        }
      }

      // Add delay between batches.
      if ($delay > 0 && $processed < $total_nodes) {
        $this->loggerSafe()->info('Waiting {delay} seconds before next batch...', ['delay' => $delay]);
        sleep((int) ceil($delay));
      }
    }

    $end_time = time();
    $duration = $end_time - $start_time;

    $this->loggerSafe()->notice('Bulk processing completed: {processed} nodes processed, {errors} errors, {duration} seconds elapsed.', [
      'processed' => $processed,
      'errors' => $errors,
      'duration' => $duration,
    ]);
  }

  /**
   * Show usage statistics.
   *
   * @command social-summaries:stats
   * @aliases ss-stats
   */
  public function stats(): void {
    $stats = $this->rateLimiter->getRateLimitStats();

    $logger = $this->loggerSafe();
    $logger->notice('Social Summaries Usage Statistics:');
    $logger->notice('  Calls in last minute: {calls}', ['calls' => $stats['calls_last_minute']]);
    $logger->notice('  Calls in last hour: {calls}', ['calls' => $stats['calls_last_hour']]);
    $logger->notice('  Calls in last day: {calls}', ['calls' => $stats['calls_last_day']]);
    $logger->notice('  Total cost today: ${cost}', ['cost' => number_format($stats['total_cost_today'], 4)]);

    // Show rate limit status.
    $can = $this->rateLimiter->canMakeCall();
    if ($can) {
      $logger->notice('Rate limit status: OK - can make API calls');
    }
    else {
      $logger->warning('Rate limit status: BLOCKED - cannot make API calls');
    }

    $slow = $this->rateLimiter->shouldSlowDown();
    if ($slow) {
      $logger->warning('Rate limit status: APPROACHING LIMIT - should slow down');
    }
  }

  /**
   * Test the AI provider connection.
   *
   * @command social-summaries:test
   * @aliases ss-test
   */
  public function test(): void {
    $config = $this->configFactory->get('social_summaries.settings');
    $provider_id = (string) ($config->get('provider') ?? 'openai');
    $provider = $this->provider ?? $this->providerManager->createInstance($provider_id);
    $result = $provider->testConnection();

    $logger = $this->loggerSafe();
    if ($result['success']) {
      $logger->notice('Connection test successful: {message}', ['message' => $result['message']]);
      if (!empty($result['details'])) {
        $logger->notice('Model: {model}, Tokens used: {tokens}', [
          'model' => $result['details']['model'],
          'tokens' => $result['details']['tokens_used'],
        ]);
      }
    }
    else {
      $logger->error('Connection test failed: {message}', ['message' => $result['message']]);
    }
  }

  /**
   * Test platform constraints being sent to AI.
   *
   * @command social-summaries:test-constraints
   * @aliases ss-test-constraints
   * @option platform The platform to test (default: linkedin)
   * @usage drush social-summaries:test-constraints --platform=linkedin
   */
  public function testConstraints(string $platform = 'linkedin'): void {
    $logger = $this->loggerSafe();
    $logger->notice('Testing constraints for platform: {platform}', ['platform' => $platform]);

    // Build messages with test content.
    $title = 'Test Article Title';
    $body = 'This is a test article body with some content to generate social media posts from.';
    $messages = $this->manager->buildMessages($title, $body, $platform);

    $logger->notice('System message: {message}', ['message' => $messages[0]['content']]);
    $logger->notice('User message: {message}', ['message' => $messages[1]['content']]);

    // Show what constraints are being sent.
    $config = $this->configFactory->get('social_summaries.settings');
    $constraint_templates = [
      'summary' => (string) ($config->get('summary_prompt_template') ?? ''),
      'linkedin' => (string) ($config->get('linkedin_prompt_template') ?? ''),
      'x' => (string) ($config->get('x_prompt_template') ?? ''),
      'newsletter' => (string) ($config->get('newsletter_prompt_template') ?? ''),
      'facebook' => (string) ($config->get('facebook_prompt_template') ?? ''),
      'threads' => (string) ($config->get('threads_prompt_template') ?? ''),
    ];

    $constraints = (string) ($constraint_templates[$platform] ?? '');
    $logger->notice('Constraints being sent to AI: {constraints}', ['constraints' => $constraints]);

    // Test actual generation.
    try {
      $provider = $this->provider ?? $this->providerManager->createInstance((string) ($this->configFactory->get('social_summaries.settings')->get('provider') ?? 'openai'));
      $resp = $provider->generate($messages);
      $text = trim($resp['text'] ?? '');
      $char_count = strlen($text);

      $logger->notice('Generated content: {content}', ['content' => $text]);
      $logger->notice('Character count: {count}', ['count' => $char_count]);

      // Check constraints.
      $constraint_data = PlatformConstraints::getConstraints($platform);
      if ($constraint_data) {
        $max_chars = $constraint_data['max_chars'] ?? 0;
        $min_chars = $constraint_data['min_chars'] ?? 0;

        if ($char_count <= $max_chars && $char_count >= $min_chars) {
          $logger->notice('PASS: Within constraints ({min}-{max} chars)', ['min' => $min_chars, 'max' => $max_chars]);
        }
        else {
          $logger->error('❌ FAIL: Outside constraints ({min}-{max} chars)', ['min' => $min_chars, 'max' => $max_chars]);
        }
      }

    }
    catch (\Exception $e) {
      $logger->error('❌ ERROR: {error}', ['error' => $e->getMessage()]);
    }
  }

  /**
   * Returns a logger that is always safe to call.
   */
  protected function loggerSafe(): LoggerInterface {
    if (method_exists($this, 'logger')) {
      $logger = $this->logger();
      if ($logger instanceof LoggerInterface) {
        return $logger;
      }
    }
    return new NullLogger();
  }

}
