<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Social Summaries bundle-specific settings.
 */
class BundleSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The node type entity.
   *
   * @var \Drupal\node\Entity\NodeType
   */
  protected $nodeType;

  /**
   * Constructs a BundleSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    // In Drupal 11, ConfigFormBase expects both configFactory and typedConfigManager in parent.
    // Use the setter to ensure config factory is available without relying on parent signature differences.
    $this->setConfigFactory($config_factory);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'social_summaries_bundle_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['social_summaries.bundle.' . $this->getNodeType()->id()];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $node_type = NULL): array {
    $this->nodeType = $this->entityTypeManager->getStorage('node_type')->load($node_type);

    if (!$this->nodeType) {
      throw new \InvalidArgumentException('Invalid node type provided.');
    }

    $config = $this->config('social_summaries.bundle.' . $this->nodeType->id());
    $global_config = $this->config('social_summaries.settings');

    // Get available platforms.
    $platforms = [
      'summary' => $this->t('Summary'),
      'linkedin' => $this->t('LinkedIn'),
      'x' => $this->t('X (Twitter)'),
      'newsletter' => $this->t('Newsletter'),
      'facebook' => $this->t('Facebook'),
      'threads' => $this->t('Threads'),
    ];

    $form['#title'] = $this->t('Social Summaries settings for @type', ['@type' => $this->nodeType->label()]);

    $form['override'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override global visibility for this content type'),
      '#description' => $this->t('When enabled, you can customize which platforms are visible for this content type. When disabled, the global settings will be used.'),
      '#default_value' => $config->get('override') ?? FALSE,
    ];

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Social Summaries for this content type'),
      '#description' => $this->t('Uncheck to disable Social Summaries entirely for this content type. The sidebar will be hidden.'),
      '#default_value' => $config->get('enabled') ?? TRUE,
    ];

    // Determine current defaults, preferring AJAX-updated values from form state.
    $current_visible_defaults = $form_state->get('visible_platforms_current');
    if (!is_array($current_visible_defaults)) {
      $current_visible_defaults = $config->get('visible_platforms') ?? ['linkedin', 'x'];
    }

    $form['visible_platforms'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Visible Platforms'),
      '#description' => $this->t('Select which platforms should be visible in the Social Summaries sidebar for this content type.'),
      '#options' => $platforms,
      '#default_value' => $current_visible_defaults,
      // Force current value during AJAX rebuilds so UI reflects select/deselect.
      '#value' => $current_visible_defaults,
      '#states' => [
        'visible' => [
          ':input[name="override"]' => ['checked' => TRUE],
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Select/Deselect all controls for visible platforms.
    $form['platform_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['social-summaries-platform-actions']],
      '#states' => [
        'visible' => [
          ':input[name="override"]' => ['checked' => TRUE],
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['platform_actions']['select_all'] = [
      '#type' => 'submit',
      '#value' => $this->t('Select all'),
      '#limit_validation_errors' => [],
      '#submit' => [[get_class($this), 'selectAllPlatforms']],
      '#ajax' => [
        'callback' => [get_class($this), 'rebuildPlatformsAjax'],
        'wrapper' => 'social-summaries-bundle-form-wrapper',
      ],
    ];
    $form['platform_actions']['deselect_all'] = [
      '#type' => 'submit',
      '#value' => $this->t('Deselect all'),
      '#limit_validation_errors' => [],
      '#submit' => [[get_class($this), 'deselectAllPlatforms']],
      '#ajax' => [
        'callback' => [get_class($this), 'rebuildPlatformsAjax'],
        'wrapper' => 'social-summaries-bundle-form-wrapper',
      ],
    ];

    // Show current global settings for reference.
    $global_platforms = $global_config->get('visible_platforms') ?? ['linkedin', 'x'];
    $global_platform_labels = array_intersect_key($platforms, array_flip($global_platforms));
    $form['global_settings_info'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['social-summaries-global-info']],
      '#states' => [
        'visible' => [
          ':input[name="override"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['global_settings_info']['info'] = [
      '#markup' => '<p><strong>' . $this->t('Current global settings:') . '</strong> ' .
      implode(', ', $global_platform_labels) . '</p>',
    ];

    // Provide a wrapper for AJAX updates of platform controls.
    $form['#prefix'] = '<div id="social-summaries-bundle-form-wrapper">';
    $form['#suffix'] = '</div>';

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
      '#button_type' => 'primary',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $override = $form_state->getValue('override');
    $visible_platforms = $form_state->getValue('visible_platforms');

    if ($override && empty(array_filter($visible_platforms))) {
      $form_state->setErrorByName('visible_platforms', $this->t('Please select at least one platform when overriding global settings.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $config = $this->configFactory->getEditable('social_summaries.bundle.' . $this->nodeType->id());

    $override = $form_state->getValue('override');
    $enabled = (bool) $form_state->getValue('enabled');
    $visible_platforms = $form_state->getValue('visible_platforms');

    $config->set('override', $override);
    $config->set('enabled', $enabled);

    if ($override) {
      // Filter out unchecked values (they come as 0).
      $config->set('visible_platforms', array_filter($visible_platforms));
    }
    else {
      // Clear bundle-specific settings when not overriding.
      $config->clear('visible_platforms');
    }

    $config->save();

    $this->messenger()->addStatus($this->t('The Social Summaries settings for @type have been saved.', ['@type' => $this->nodeType->label()]));
  }

  /**
   * AJAX callback to rebuild form section.
   */
  public static function rebuildPlatformsAjax(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * Submit handler: Select all platforms.
   */
  public static function selectAllPlatforms(array &$form, FormStateInterface $form_state) {
    $platforms = array_keys((array) $form['visible_platforms']['#options']);
    $values = array_combine($platforms, $platforms);
    // Update both internal state and user input so FAPI uses these values.
    $form_state->set('visible_platforms_current', $values);
    $form_state->setValue('visible_platforms', $values);
    $input = (array) $form_state->getUserInput();
    $input['visible_platforms'] = $values;
    $form_state->setUserInput($input);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit handler: Deselect all platforms.
   */
  public static function deselectAllPlatforms(array &$form, FormStateInterface $form_state) {
    // For checkboxes, unselected options should be absent from user input.
    $form_state->set('visible_platforms_current', []);
    $form_state->setValue('visible_platforms', []);
    $input = (array) $form_state->getUserInput();
    $input['visible_platforms'] = [];
    $form_state->setUserInput($input);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Gets the node type entity.
   *
   * @return \Drupal\node\Entity\NodeType
   *   The node type entity.
   */
  protected function getNodeType() {
    return $this->nodeType;
  }

}
