<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Generator;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;

/**
 * Builds prompts and logs token usage.
 */
class GenerationManager {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $db;

  /**
   * Constructs a GenerationManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Database\Connection $db
   *   The database connection.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger, Connection $db) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->db = $db;
  }

  /**
   * Build messages for AI generation.
   *
   * @param string $title
   *   The content title.
   * @param string $body
   *   The content body.
   * @param string $platform
   *   The generation platform.
   * @param array $vars
   *   Additional variables.
   *
   * @return array
   *   The messages array.
   */
  public function buildMessages(string $title, string $body, string $platform = 'summary', array $vars = []): array {
    $config = $this->configFactory->get('social_summaries.settings');
    $tone = $vars['tone'] ?? $config->get('tone') ?? 'neutral, professional';

    // Get customizable prompt templates from configuration.
    $system_template = (string) ($config->get('system_prompt_template') ?? '');
    $constraints_map = (array) ($config->get('constraints') ?? []);
    $constraint_templates = [
      'summary' => (string) (($constraints_map['summary']['prompt_template'] ?? '')),
      'linkedin' => (string) (($constraints_map['linkedin']['prompt_template'] ?? '')),
      'x' => (string) (($constraints_map['x']['prompt_template'] ?? '')),
      'newsletter' => (string) (($constraints_map['newsletter']['prompt_template'] ?? '')),
      'facebook' => (string) (($constraints_map['facebook']['prompt_template'] ?? '')),
      'threads' => (string) (($constraints_map['threads']['prompt_template'] ?? '')),
    ];

    // Build a human-readable constraints string from saved numeric limits.
    $all_numeric_constraints = (array) ($config->get('constraints') ?? []);
    $numeric_constraints = (array) ($all_numeric_constraints[$platform] ?? []);
    $numeric_text = $this->formatNumericConstraints($numeric_constraints);

    // Combine per-platform prompt template text (if any) and numeric constraint text.
    $constraints = trim(implode(' ', array_filter([
      (string) ($constraint_templates[$platform] ?? ''),
      $numeric_text,
    ])));

    // Replace placeholders in system prompt.
    $system_content = str_replace(['{tone}', '{constraints}'], [$tone, $constraints], $system_template);
    $system = [
      'role' => 'system',
      'content' => $system_content,
    ];
    $user = [
      'role' => 'user',
      'content' => "Title: {$title}\n\nBody:\n" . mb_substr($body, 0, 4000) . "\n\nPlatform: {$platform}",
    ];
    return [$system, $user];
  }

  /**
   * Formats numeric constraint limits into a human-readable string.
   *
   * @param array $c
   *   The numeric constraints for a platform.
   *
   * @return string
   *   A sentence-like description of limits and recommendations.
   */
  protected function formatNumericConstraints(array $c): string {
    $parts = [];
    // Character limits.
    if (isset($c['min_chars']) || isset($c['max_chars'])) {
      if (isset($c['min_chars']) && isset($c['max_chars'])) {
        $parts[] = sprintf('%d-%d characters', (int) $c['min_chars'], (int) $c['max_chars']);
      }
      elseif (isset($c['max_chars'])) {
        $parts[] = sprintf('Maximum %d characters', (int) $c['max_chars']);
      }
    }
    // Word limits.
    if (isset($c['min_words']) || isset($c['max_words'])) {
      if (isset($c['min_words']) && isset($c['max_words'])) {
        $parts[] = sprintf('%d-%d words', (int) $c['min_words'], (int) $c['max_words']);
      }
      elseif (isset($c['max_words'])) {
        $parts[] = sprintf('Maximum %d words', (int) $c['max_words']);
      }
    }
    // Sentence limits.
    if (isset($c['min_sentences']) || isset($c['max_sentences'])) {
      if (isset($c['min_sentences']) && isset($c['max_sentences'])) {
        $parts[] = sprintf('%d-%d sentences', (int) $c['min_sentences'], (int) $c['max_sentences']);
      }
      elseif (isset($c['max_sentences'])) {
        $parts[] = sprintf('Maximum %d sentences', (int) $c['max_sentences']);
      }
    }
    // Recommended ranges (optional).
    if (isset($c['recommended_min']) && isset($c['recommended_max'])) {
      $parts[] = sprintf('Recommended %d-%d', (int) $c['recommended_min'], (int) $c['recommended_max']);
    }

    return trim(implode(', ', $parts));
  }

  /**
   * Estimate cost in USD for token usage.
   *
   * @param int $tokens_in
   *   Input tokens.
   * @param int $tokens_out
   *   Output tokens.
   * @param string $model
   *   The model name.
   *
   * @return float
   *   The estimated cost in USD.
   */
  public function estimateCostUsd(int $tokens_in, int $tokens_out, string $model = 'gpt-4.1-mini'): float {
    // OpenAI pricing per 1K tokens (as of 2024)
    $pricing = [
      'gpt-4o' => ['input' => 0.005, 'output' => 0.015],
      'gpt-4o-mini' => ['input' => 0.00015, 'output' => 0.0006],
      // Same pricing as gpt-4o-mini.
      'gpt-4.1-mini' => ['input' => 0.00015, 'output' => 0.0006],
      'gpt-3.5-turbo' => ['input' => 0.0005, 'output' => 0.0015],
    ];

    $rates = $pricing[$model] ?? $pricing['gpt-4.1-mini'];
    $input_cost = ($tokens_in / 1000) * $rates['input'];
    $output_cost = ($tokens_out / 1000) * $rates['output'];
    return round($input_cost + $output_cost, 6);
  }

  /**
   * Log token usage to database.
   *
   * @param int|null $nid
   *   The node ID.
   * @param string $platform
   *   The generation platform.
   * @param string $model
   *   The model name.
   * @param int $tokens_in
   *   Input tokens.
   * @param int $tokens_out
   *   Output tokens.
   */
  public function logUsage(?int $nid, string $platform, string $model, int $tokens_in, int $tokens_out): void {
    $cost = $this->estimateCostUsd($tokens_in, $tokens_out, $model);
    $this->db->insert('social_summaries_usage')
      ->fields([
        'nid' => $nid,
        'platform' => $platform,
        'model' => $model,
        'tokens_in' => $tokens_in,
        'tokens_out' => $tokens_out,
        'cost_usd' => $cost,
        'created' => time(),
      ])
      ->execute();
  }

}
