<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Plugin\SocialSummariesProvider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\social_summaries\Provider\OpenAiProvider as LegacyOpenAiProvider;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * OpenAI provider plugin for Social Summaries.
 *
 * @SocialSummariesProvider(
 *   id = "openai",
 *   label = @Translation("OpenAI"),
 *   description = @Translation("Generate content using OpenAI's GPT models."),
 *   api_documentation = "https://platform.openai.com/docs/api-reference",
 *   requires_api_key = TRUE,
 *   default_model = "gpt-4o-mini",
 *   max_tokens = 4096,
 *   default_temperature = 0.5
 * )
 */
class OpenAiProvider extends SocialSummariesProviderBase {

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs an OpenAiProvider object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel for this provider.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(array $configuration, string $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, KeyRepositoryInterface $key_repository, ClientInterface $http_client, LoggerInterface $logger, TranslationInterface $string_translation) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $config_factory, $string_translation);
    $this->keyRepository = $key_repository;
    $this->httpClient = $http_client;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('http_client'),
      $container->get('logger.factory')->get('social_summaries'),
      $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedModels(): array {
    return [
      'gpt-4o' => 'GPT-4o',
      'gpt-4o-mini' => 'GPT-4o Mini',
      'gpt-4-turbo' => 'GPT-4 Turbo',
      'gpt-4' => 'GPT-4',
      'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getRateLimits(): array {
    return [
      'requests_per_minute' => 60,
      'requests_per_hour' => 1000,
      'tokens_per_minute' => 150000,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateCost(int $tokens_in, int $tokens_out, string $model): float {
    // OpenAI pricing as of 2024 (per 1M tokens)
    $pricing = [
      'gpt-4o' => ['input' => 5.00, 'output' => 15.00],
      'gpt-4o-mini' => ['input' => 0.15, 'output' => 0.60],
      'gpt-4-turbo' => ['input' => 10.00, 'output' => 30.00],
      'gpt-4' => ['input' => 30.00, 'output' => 60.00],
      'gpt-3.5-turbo' => ['input' => 0.50, 'output' => 1.50],
    ];

    if (!isset($pricing[$model])) {
      return 0.0;
    }

    $input_cost = ($tokens_in / 1000000) * $pricing[$model]['input'];
    $output_cost = ($tokens_out / 1000000) * $pricing[$model]['output'];

    return $input_cost + $output_cost;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $config = $this->configFactory->get('social_summaries.settings');

    // Get available keys for selection.
    $keys = $this->keyRepository->getKeys();
    $key_options = ['' => $this->t('- Select a key -')];
    foreach ($keys as $key_id => $key) {
      $key_options[$key_id] = $key->label();
    }

    $form['api_key'] = [
      '#type' => 'select',
      '#title' => $this->t('OpenAI API Key'),
      '#options' => $key_options,
      '#default_value' => $config->get('openai_api_key_id') ?? '',
      '#description' => $this->t('Select a key from the Key module. Create a new key at <a href="@url">Configuration > System > Key</a> if needed.', [
        '@url' => '/admin/config/system/key',
      ]),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function generate(array $messages, array $options = []): array {
    $config = $this->configFactory->get('social_summaries.settings');

    // Get API key.
    $api_key_id = $config->get('openai_api_key_id');
    if (empty($api_key_id)) {
      throw new \RuntimeException('OpenAI API key not configured.');
    }

    $key = $this->keyRepository->getKey($api_key_id);
    if (!$key) {
      throw new \RuntimeException('OpenAI API key not found.');
    }

    $api_key = $key->getKeyValue();
    if (empty($api_key)) {
      throw new \RuntimeException('OpenAI API key is empty.');
    }

    // Use the legacy provider for actual generation (reuse its robust logic)
    $legacy_provider = new LegacyOpenAiProvider($this->httpClient, $this->configFactory, $this->keyRepository, $this->logger);
    return $legacy_provider->generate($messages);
  }

  /**
   * {@inheritdoc}
   */
  public function testConnection(): array {
    try {
      $result = $this->generate([
        [
          'role' => 'user',
          'content' => 'Hello, this is a test message. Please respond with "Connection successful."',
        ],
      ]);

      return [
        'success' => TRUE,
        'message' => $this->t('Connection to OpenAI successful.'),
        'details' => [
          'model' => $result['model'] ?? 'unknown',
          'tokens_used' => ($result['usage']['tokens_in'] ?? 0) + ($result['usage']['tokens_out'] ?? 0),
        ],
      ];
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => $this->t('Connection failed: @error', ['@error' => $e->getMessage()]),
      ];
    }
  }

}
