<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Plugin\SocialSummariesProvider;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * Defines the interface for Social Summaries provider plugins.
 */
interface SocialSummariesProviderInterface extends PluginInspectionInterface {

  /**
   * Generates content using the AI provider.
   *
   * @param array $messages
   *   The messages to send to the AI provider.
   * @param array $options
   *   Additional options for generation (model, temperature, etc.).
   *
   * @return array
   *   An array containing:
   *   - text: The generated text
   *   - usage: Array with tokens_in, tokens_out
   *   - model: The model used
   *
   * @throws \Exception
   *   If generation fails.
   */
  public function generate(array $messages, array $options = []): array;

  /**
   * Tests the connection to the AI provider.
   *
   * @return array
   *   An array containing:
   *   - success: Boolean indicating if the test was successful
   *   - message: Human-readable message about the test result
   *   - details: Optional array with additional details (model, tokens_used, etc.)
   */
  public function testConnection(): array;

  /**
   * Gets the provider's configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array
   *   The form array with provider-specific configuration.
   */
  public function buildConfigurationForm(array $form, $form_state): array;

  /**
   * Validates the provider's configuration.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateConfigurationForm(array $form, $form_state): void;

  /**
   * Submits the provider's configuration.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function submitConfigurationForm(array $form, $form_state): void;

  /**
   * Gets the provider's default configuration.
   *
   * @return array
   *   The default configuration array.
   */
  public function getDefaultConfiguration(): array;

  /**
   * Gets the provider's supported models.
   *
   * @return array
   *   An array of supported models keyed by model ID.
   */
  public function getSupportedModels(): array;

  /**
   * Gets the provider's rate limits.
   *
   * @return array
   *   An array containing rate limit information:
   *   - requests_per_minute: Maximum requests per minute
   *   - requests_per_hour: Maximum requests per hour
   *   - tokens_per_minute: Maximum tokens per minute
   */
  public function getRateLimits(): array;

  /**
   * Calculates the cost for a given number of tokens.
   *
   * @param int $tokens_in
   *   Number of input tokens.
   * @param int $tokens_out
   *   Number of output tokens.
   * @param string $model
   *   The model used.
   *
   * @return float
   *   The cost in USD.
   */
  public function calculateCost(int $tokens_in, int $tokens_out, string $model): float;

}
