# Contributing to Social Summaries

Thank you for your interest in contributing to the Social Summaries module! This document provides guidelines and information for contributors.

## Table of Contents

- [Code of Conduct](#code-of-conduct)
- [Getting Started](#getting-started)
- [Development Setup](#development-setup)
- [Coding Standards](#coding-standards)
- [Testing](#testing)
- [Pull Request Process](#pull-request-process)
- [Issue Reporting](#issue-reporting)
- [Documentation](#documentation)

## Code of Conduct

This project follows the [Drupal Code of Conduct](https://www.drupal.org/dcoc). Please read and follow these guidelines in all interactions.

## Getting Started

### Prerequisites

- Drupal 10.x or 11.x
- PHP 8.1 or higher
- Composer
- Drush
- Git

### Development Setup

1. **Fork the repository** on Drupal.org or GitHub
2. **Clone your fork**:
   ```bash
   git clone https://git.drupalcode.org/issue/YOUR-USERNAME/social_summaries.git
   cd social_summaries
   ```
3. **Install dependencies**:
   ```bash
   composer install
   ```
4. **Set up a Drupal site** for testing
5. **Enable the module**:
   ```bash
   drush en social_summaries
   ```

## Coding Standards

### PHP Standards

- Follow [Drupal Coding Standards](https://www.drupal.org/docs/develop/standards)
- Use PHP 8.1+ features appropriately
- Include proper type hints and return types
- Add comprehensive docblocks for all public methods
- Use strict typing: `declare(strict_types=1);`

### Code Style

Run the following commands to check and fix code style:

```bash
# Check coding standards
vendor/bin/phpcs --standard=Drupal,DrupalPractice src/ tests/

# Fix auto-fixable issues
vendor/bin/phpcbf --standard=Drupal,DrupalPractice src/ tests/

# Run PHPStan static analysis
vendor/bin/phpstan analyse src/
```

### JavaScript Standards

- Follow [Drupal JavaScript Coding Standards](https://www.drupal.org/docs/develop/standards/javascript)
- Use modern ES6+ features
- Include proper JSDoc comments
- Ensure accessibility compliance

### CSS Standards

- Follow [Drupal CSS Coding Standards](https://www.drupal.org/docs/develop/standards/css)
- Use BEM methodology for class naming
- Ensure responsive design
- Maintain accessibility standards

## Testing

### Test Requirements

All contributions must include appropriate tests:

- **Unit Tests**: For individual classes and methods
- **Kernel Tests**: For integration with Drupal services
- **Functional Tests**: For end-to-end user workflows
- **Accessibility Tests**: For WCAG 2.1 AA compliance

### Running Tests

```bash
# Run all tests
./run-tests.sh

# Run specific test suites
vendor/bin/phpunit tests/src/Unit/
vendor/bin/phpunit tests/src/Kernel/
vendor/bin/phpunit tests/src/Functional/

# Run with coverage
vendor/bin/phpunit --coverage-html coverage/
```

### Test Coverage

- Maintain 90%+ code coverage
- All new features must have corresponding tests
- Bug fixes must include regression tests

### Accessibility Testing

```bash
# Run accessibility tests
vendor/bin/phpunit tests/src/Functional/SocialSummariesAccessibilityTest.php
vendor/bin/phpunit tests/src/Functional/SocialSummariesJavaScriptAccessibilityTest.php
```

## Pull Request Process

### Before Submitting

1. **Create a feature branch** from the latest main branch
2. **Write tests** for your changes
3. **Update documentation** as needed
4. **Run all tests** and ensure they pass
5. **Check coding standards** compliance
6. **Test manually** in a Drupal environment

### Pull Request Checklist

- [ ] Code follows Drupal coding standards
- [ ] All tests pass (unit, kernel, functional, accessibility)
- [ ] Test coverage is maintained or improved
- [ ] Documentation is updated
- [ ] CHANGELOG.md is updated (if applicable)
- [ ] No breaking changes (or clearly documented)
- [ ] Manual testing completed
- [ ] Security implications considered

### Pull Request Template

```markdown
## Description
Brief description of changes

## Type of Change
- [ ] Bug fix
- [ ] New feature
- [ ] Breaking change
- [ ] Documentation update

## Testing
- [ ] Unit tests added/updated
- [ ] Kernel tests added/updated
- [ ] Functional tests added/updated
- [ ] Accessibility tests pass
- [ ] Manual testing completed

## Checklist
- [ ] Code follows Drupal standards
- [ ] Self-review completed
- [ ] Documentation updated
- [ ] CHANGELOG.md updated
```

## Issue Reporting

### Bug Reports

When reporting bugs, please include:

- **Drupal version** and **PHP version**
- **Module version**
- **Steps to reproduce**
- **Expected behavior**
- **Actual behavior**
- **Error messages** (if any)
- **Screenshots** (if applicable)

### Feature Requests

For feature requests, please include:

- **Use case** and **justification**
- **Proposed solution**
- **Alternative solutions** considered
- **Impact** on existing functionality

### Security Issues

**Do not** report security issues publicly. Instead:

1. **Email** security issues to the maintainer privately
2. **Include** detailed reproduction steps
3. **Wait** for acknowledgment before public disclosure

## Documentation

### Code Documentation

- **PHPDoc** for all public methods and classes
- **Inline comments** for complex logic
- **README updates** for new features
- **API documentation** for new services

### Adding New AI Providers

To add support for a new AI provider:

1. **Create a provider plugin** in `src/Plugin/SocialSummariesProvider/`
2. **Implement the interface** `SocialSummariesProviderInterface`
3. **Add the annotation** with provider metadata
4. **Register the plugin** in your module's services
5. **Add tests** for the new provider
6. **Update documentation** with provider-specific information

Example provider plugin structure:
```php
/**
 * @SocialSummariesProvider(
 *   id = "your_provider",
 *   label = @Translation("Your Provider"),
 *   description = @Translation("Description of your provider"),
 *   api_documentation = "https://your-provider.com/docs",
 *   requires_api_key = TRUE,
 *   default_model = "your-model",
 *   max_tokens = 4096,
 *   default_temperature = 0.5
 * )
 */
class YourProvider extends SocialSummariesProviderBase {
  // Implement required methods
}
```

### User Documentation

- **README.md** updates for new features
- **Configuration** documentation
- **Usage examples** and **screenshots**
- **Troubleshooting** guides

## Development Workflow

### Branch Naming

- `feature/description` - New features
- `bugfix/description` - Bug fixes
- `hotfix/description` - Critical fixes
- `docs/description` - Documentation updates

### Commit Messages

Follow [Conventional Commits](https://www.conventionalcommits.org/):

```
feat: add new platform support
fix: resolve rate limiting issue
docs: update installation instructions
test: add accessibility test coverage
```

### Release Process

1. **Update CHANGELOG.md** with new version
2. **Update version** in info.yml and composer.json
3. **Create release tag**
4. **Update documentation**
5. **Announce release**

## Getting Help

- **Issue Queue**: [Drupal.org](https://www.drupal.org/project/issues/social_summaries)
- **Documentation**: See README.md and TESTING.md
- **Community**: Drupal Slack #contribute channel

## Recognition

Contributors will be recognized in:
- **CHANGELOG.md** for significant contributions
- **README.md** contributors section
- **Drupal.org** project page

Thank you for contributing to Social Summaries! 🚀
